package org.tio.http.server.handler;

import java.beans.PropertyDescriptor;
import java.io.File;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ExecutionException;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.Aio;
import org.tio.core.ChannelContext;
import org.tio.http.common.Cookie;
import org.tio.http.common.HttpConfig;
import org.tio.http.common.HttpConst;
import org.tio.http.common.HttpRequest;
import org.tio.http.common.HttpResponse;
import org.tio.http.common.HttpResponseStatus;
import org.tio.http.common.RequestLine;
import org.tio.http.common.handler.HttpRequestHandler;
import org.tio.http.common.session.HttpSession;
import org.tio.http.common.utils.IpUtils;
import org.tio.http.server.intf.HttpServerInterceptor;
import org.tio.http.server.intf.HttpSessionListener;
import org.tio.http.server.intf.ThrowableHandler;
import org.tio.http.server.mvc.Routes;
import org.tio.http.server.session.SessionCookieDecorator;
import org.tio.http.server.stat.ip.path.IpAccessStat;
import org.tio.http.server.stat.ip.path.IpPathAccessStat;
import org.tio.http.server.stat.ip.path.IpPathAccessStatListener;
import org.tio.http.server.stat.ip.path.IpPathAccessStats;
import org.tio.http.server.util.ClassUtils;
import org.tio.http.server.util.Resps;
import org.tio.http.server.view.freemarker.FreemarkerConfig;
import org.tio.utils.SystemTimer;
import org.tio.utils.cache.guava.GuavaCache;
import org.tio.utils.freemarker.FreemarkerUtils;

import com.xiaoleilu.hutool.bean.BeanUtil;
import com.xiaoleilu.hutool.convert.Convert;
import com.xiaoleilu.hutool.util.ArrayUtil;
import com.xiaoleilu.hutool.util.ClassUtil;
import com.xiaoleilu.hutool.util.ZipUtil;

import freemarker.cache.FileTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import jodd.io.FileNameUtil;

/**
 *
 * @author tanyaowu
 *
 */
public class DefaultHttpRequestHandler implements HttpRequestHandler {
	private static Logger log = LoggerFactory.getLogger(DefaultHttpRequestHandler.class);

	//	/**
	//	 * 静态资源的CacheName
	//	 * key:   path 譬如"/index.html"
	//	 * value: HttpResponse
	//	 */
	//	private static final String STATIC_RES_CACHENAME = "TIO_HTTP_STATIC_RES";

	/**
	 * 静态资源的CacheName
	 * key:   path 譬如"/index.html"
	 * value: FileCache
	 */
	private static final String STATIC_RES_CONTENT_CACHENAME = "TIO_HTTP_STATIC_RES_CONTENT";

	/**
	 * @param args
	 *
	 * @author tanyaowu
	 * 2016年11月18日 上午9:13:15
	 *
	 */
	public static void main(String[] args) {
	}

	protected HttpConfig httpConfig;

	protected Routes routes = null;

	//	private LoadingCache<String, HttpSession> loadingCache = null;

	private HttpServerInterceptor httpServerInterceptor;

	private HttpSessionListener httpSessionListener;
	
	private ThrowableHandler throwableHandler;

	private SessionCookieDecorator sessionCookieDecorator;

	private IpPathAccessStats ipPathAccessStats;

	private GuavaCache staticResCache;

	private String contextPath;
	private int contextPathLength = 0;
	private String suffix;
	private int suffixLength = 0;

	/**
	 * 临时支持freemarker，主要用于开发环境中的前端开发，暂时不重点作为tio-http-server功能
	 */
	private FreemarkerConfig freemarkerConfig;

	//	private static String randomCookieValue() {
	//		return RandomUtil.randomUUID();
	//	}

	/**
	 *
	 * @param httpConfig
	 * @param routes
	 * @author tanyaowu
	 */
	public DefaultHttpRequestHandler(HttpConfig httpConfig, Routes routes) {
		if (httpConfig == null) {
			throw new RuntimeException("httpConfig can not be null");
		}
		this.contextPath = httpConfig.getContextPath();
		this.suffix = httpConfig.getSuffix();

		if (StringUtils.isNotBlank(contextPath)) {
			contextPathLength = contextPath.length();
		}
		if (StringUtils.isNotBlank(suffix)) {
			suffixLength = suffix.length();
		}

		this.httpConfig = httpConfig;

		if (httpConfig.getMaxLiveTimeOfStaticRes() > 0) {
			staticResCache = GuavaCache.register(STATIC_RES_CONTENT_CACHENAME, (long) httpConfig.getMaxLiveTimeOfStaticRes(), null);
		}

		this.routes = routes;
	}

	/**
	 * 创建httpsession
	 * @return
	 * @author tanyaowu
	 */
	private HttpSession createSession(HttpRequest request) {
		String sessionId = httpConfig.getSessionIdGenerator().sessionId(httpConfig, request);
		HttpSession httpSession = new HttpSession(sessionId);
		if (httpSessionListener != null) {
			httpSessionListener.doAfterCreated(request, httpSession, httpConfig);
		}
		return httpSession;
	}

	/**
	 * @return the httpConfig
	 */
	public HttpConfig getHttpConfig() {
		return httpConfig;
	}

	public HttpServerInterceptor getHttpServerInterceptor() {
		return httpServerInterceptor;
	}

	public static Cookie getSessionCookie(HttpRequest request, HttpConfig httpConfig) {
		Cookie sessionCookie = request.getCookie(httpConfig.getSessionCookieName());
		return sessionCookie;
	}

	/**
	 * @return the staticResCache
	 */
	public GuavaCache getStaticResCache() {
		return staticResCache;
	}

	/**
	 * 检查域名是否可以访问本站
	 * @param request
	 * @return
	 * @author tanyaowu
	 */
	private boolean checkDomain(HttpRequest request) {
		String[] allowDomains = httpConfig.getAllowDomains();
		if (allowDomains == null || allowDomains.length == 0) {
			return true;
		}
		String host = request.getHost();
		if (ArrayUtil.contains(allowDomains, host)) {
			return true;
		}
		return false;
	}
	
	/**
	 * 
	 * @param request
	 * @param response
	 * @author tanyaowu
	 */
	private static void gzip(HttpRequest request, HttpResponse response) {
		if (response == null) {
			return;
		}
		
		if (request.getIsSupportGzip()) {
			byte[] bs = response.getBody();
			if (bs != null && bs.length >= 600) {
				byte[] bs2 = ZipUtil.gzip(bs);
				if (bs2.length < bs.length) {
					response.setBody(bs2, request);
					response.addHeader(HttpConst.ResponseHeaderKey.Content_Encoding, "gzip");
				}
			}
		} else {
			log.info("{} 竟然不支持gzip, {}", request.getChannelContext(), request.getHeader(HttpConst.RequestHeaderKey.User_Agent));
		}
	}

	@Override
	public HttpResponse handler(HttpRequest request) throws Exception {
		if (!checkDomain(request)) {
			Aio.remove(request.getChannelContext(), "过来的域名[" + request.getDomain() + "]不对");
			return null;
		}

		HttpResponse ret = null;
		RequestLine requestLine = request.getRequestLine();
		String path = requestLine.getPath();

		if (StringUtils.isNotBlank(contextPath)) {
			if (StringUtils.startsWith(path, contextPath)) {
				path = StringUtils.substring(path, contextPathLength);
			} else {
				//				Aio.remove(request.getChannelContext(), "请求路径不合法，必须以" + contextPath + "开头：" + requestLine.getLine());
				//				return null;
			}
		}

		if (StringUtils.isNotBlank(suffix)) {
			if (StringUtils.endsWith(path, suffix)) {
				path = StringUtils.substring(path, 0, path.length() - suffixLength);
			} else {
				//				Aio.remove(request.getChannelContext(), "请求路径不合法，必须以" + suffix + "结尾：" + requestLine.getLine());
				//				return null;
			}
		}
		requestLine.setPath(path);

		if (ipPathAccessStats != null) {
			String ip = IpUtils.getRealIp(request);
			List<Long> list = ipPathAccessStats.durationList;

			Cookie cookie = getSessionCookie(request, httpConfig);

			for (Long duration : list) {
				IpAccessStat ipAccessStat = ipPathAccessStats.get(duration, ip);//.get(duration, ip, path);//.get(v, channelContext.getClientNode().getIp());

				ipAccessStat.count.incrementAndGet();
				ipAccessStat.setLastAccessTime(SystemTimer.currentTimeMillis());

				IpPathAccessStat ipPathAccessStat = ipAccessStat.get(path);
				ipPathAccessStat.count.incrementAndGet();
				ipPathAccessStat.setLastAccessTime(SystemTimer.currentTimeMillis());

				if (cookie == null) {
					ipAccessStat.noSessionCount.incrementAndGet();
					ipPathAccessStat.noSessionCount.incrementAndGet();
				} else {
					ipAccessStat.sessionIds.add(cookie.getValue());
				}

				IpPathAccessStatListener ipPathAccessStatListener = ipPathAccessStats.getListener(duration);
				if (ipPathAccessStatListener != null) {
					boolean isContinue = ipPathAccessStatListener.onChanged(request, ip, path, ipAccessStat, ipPathAccessStat);
					if (!isContinue) {
						return null;
					}
				}
			}
		}

		try {

			processCookieBeforeHandler(request, requestLine);
			HttpSession httpSession = request.getHttpSession();//(HttpSession) channelContext.getAttribute();

			//			GuavaCache guavaCache = GuavaCache.getCache(STATIC_RES_CACHENAME);
			//			ret = (HttpResponse) guavaCache.get(requestLine.getPath());
			//			if (ret != null) {
			//				log.info("从缓存中获取响应:{}", requestLine.getPath());
			//			}

			if (httpServerInterceptor != null) {
				ret = httpServerInterceptor.doBeforeHandler(request, requestLine, ret);
				if (ret != null) {
					return ret;
				}
			}
			requestLine = request.getRequestLine();
			//			if (ret != null) {
			//				return ret;
			//			}

			path = requestLine.getPath();

			Method method = null;
			if (routes != null) {
				method = routes.pathMethodMap.get(path);
			}

			if (method != null) {
				String[] paramnames = routes.methodParamnameMap.get(method);
				Class<?>[] parameterTypes = method.getParameterTypes();

				Object bean = routes.methodBeanMap.get(method);
				Object obj = null;
				Map<String, Object[]> params = request.getParams();
				if (parameterTypes == null || parameterTypes.length == 0) {
					obj = method.invoke(bean);
				} else {
					//赋值这段代码待重构，先用上
					Object[] paramValues = new Object[parameterTypes.length];
					int i = 0;
					for (Class<?> paramType : parameterTypes) {
						try {
							if (paramType.isAssignableFrom(HttpRequest.class)) {
								paramValues[i] = request;
							} else if (paramType == HttpSession.class) {
								paramValues[i] = httpSession;
							} else if (paramType.isAssignableFrom(HttpConfig.class)) {
								paramValues[i] = httpConfig;
							} else if (paramType.isAssignableFrom(ChannelContext.class)) {
								paramValues[i] = request.getChannelContext();
							} else {
								if (params != null) {
									if (ClassUtils.isSimpleTypeOrArray(paramType)) {
										//										paramValues[i] = Ognl.getValue(paramnames[i], (Object) params, paramType);
										Object[] value = params.get(paramnames[i]);
										if (value != null && value.length > 0) {
											if (paramType.isArray()) {
												paramValues[i] = Convert.convert(paramType, value);
											} else {
												paramValues[i] = Convert.convert(paramType, value[0]);
											}
										}
									} else {
										paramValues[i] = paramType.newInstance();//BeanUtil.mapToBean(params, paramType, true);
										Set<Entry<String, Object[]>> set = params.entrySet();
										label2: for (Entry<String, Object[]> entry : set) {
											String fieldName = entry.getKey();
											Object[] fieldValue = entry.getValue();

											PropertyDescriptor propertyDescriptor = BeanUtil.getPropertyDescriptor(paramType, fieldName, true);
											if (propertyDescriptor == null) {
												continue label2;
											} else {
												Method writeMethod = propertyDescriptor.getWriteMethod();
												if (writeMethod == null) {
													continue label2;
												}
												writeMethod = ClassUtil.setAccessible(writeMethod);
												Class<?>[] clazzes = writeMethod.getParameterTypes();
												if (clazzes == null || clazzes.length != 1) {
													log.info("方法的参数长度不为1，{}.{}", paramType.getName(), writeMethod.getName());
													continue label2;
												}
												Class<?> clazz = clazzes[0];

												if (ClassUtils.isSimpleTypeOrArray(clazz)) {
													if (fieldValue != null && fieldValue.length > 0) {
														if (clazz.isArray()) {
															writeMethod.invoke(paramValues[i], Convert.convert(clazz, fieldValue));
														} else {
															writeMethod.invoke(paramValues[i], Convert.convert(clazz, fieldValue[0]));
														}
													}
												}
											}
										}
									}
								}
							}
						} catch (Throwable e) {
							log.error(e.toString(), e);
						} finally {
							i++;
						}
					}
					obj = method.invoke(bean, paramValues);
				}

				if (obj instanceof HttpResponse) {
					ret = (HttpResponse) obj;
					return ret;
				} else {
					if (obj == null) {
						//ret = Resps.txt(request, "");//.json(request, obj + "");
						return null;
					} else {
						ret = Resps.json(request, obj);
					}
					return ret;
				}
			} else {
				GuavaCache contentCache = null;
				FileCache fileCache = null;
				if (httpConfig.getMaxLiveTimeOfStaticRes() > 0) {
					contentCache = GuavaCache.getCache(STATIC_RES_CONTENT_CACHENAME);
					fileCache = (FileCache) contentCache.get(path);
				}
				if (fileCache != null) {
					byte[] bodyBytes = fileCache.getData();
					Map<String, String> headers = fileCache.getHeaders();
					long lastModified = fileCache.getLastModified();
					log.info("从缓存获取:[{}], {}", path, bodyBytes.length);

					ret = Resps.try304(request, lastModified);
					if (ret != null) {
						ret.addHeader(HttpConst.ResponseHeaderKey.tio_from_cache, "true");

						return ret;
					}

					ret = new HttpResponse(request, httpConfig);
					ret.setBody(bodyBytes, request);
					ret.addHeaders(headers);
					return ret;
				} else {
					File pageRoot = httpConfig.getPageRoot();
					if (pageRoot != null) {
						//						String root = FileUtil.getAbsolutePath(pageRoot);
						File file = new File(pageRoot + path);
						if (!file.exists() || file.isDirectory()) {
							if (StringUtils.endsWith(path, "/")) {
								path = path + "index.html";
							} else {
								path = path + "/index.html";
							}
							file = new File(pageRoot, path);
						}

						if (file.exists()) {
							if (freemarkerConfig != null) {
								String extension = FileNameUtil.getExtension(file.getName());
								if (ArrayUtil.contains(freemarkerConfig.getSuffixes(), extension)) {
									Configuration configuration = freemarkerConfig.getConfiguration();
									Object model = freemarkerConfig.getModelMaker().maker(request);
									if (configuration != null) {
										TemplateLoader templateLoader = configuration.getTemplateLoader();//FileTemplateLoader
										if (templateLoader instanceof FileTemplateLoader) {
											try {
												String filePath = file.getCanonicalPath();
												String pageRootPath = httpConfig.getPageRoot().getCanonicalPath();
												String template = StringUtils.substring(filePath, pageRootPath.length());
												String retStr = FreemarkerUtils.generateStringByFile(template, configuration, model);
												ret = Resps.file(request, retStr.getBytes(configuration.getDefaultEncoding()), extension);
												return ret;
											} catch (Exception e) {
												//freemarker编译异常的全部走普通view
												log.error(e.toString());
											}
										}
									}
								}
							}

							ret = Resps.file(request, file);
							ret.setStaticRes(true);

							if (contentCache != null && request.getIsSupportGzip()) {
								if (ret.getBody() != null && ret.getStatus() == HttpResponseStatus.C200) {
									String contentType = ret.getHeader(HttpConst.ResponseHeaderKey.Content_Type);
									String contentEncoding = ret.getHeader(HttpConst.ResponseHeaderKey.Content_Encoding);
									String lastModified = ret.getHeader(HttpConst.ResponseHeaderKey.Last_Modified);

									Map<String, String> headers = new HashMap<>();
									if (StringUtils.isNotBlank(contentType)) {
										headers.put(HttpConst.ResponseHeaderKey.Content_Type, contentType);
									}
									if (StringUtils.isNotBlank(contentEncoding)) {
										headers.put(HttpConst.ResponseHeaderKey.Content_Encoding, contentEncoding);
									}
									if (StringUtils.isNotBlank(lastModified)) {
										headers.put(HttpConst.ResponseHeaderKey.Last_Modified, lastModified);
									}
									headers.put(HttpConst.ResponseHeaderKey.tio_from_cache, "true");

									fileCache = new FileCache(headers, file.lastModified(), ret.getBody());
									contentCache.put(path, fileCache);
									log.info("放入缓存:[{}], {}", path, ret.getBody().length);
								}
							}

							return ret;
						}
					}
				}
			}

			ret = resp404(request, requestLine);//Resps.html(request, "404--并没有找到你想要的内容", httpConfig.getCharset());
			return ret;
		} catch (Throwable e) {
			logError(request, requestLine, e);
			ret = resp500(request, requestLine, e);//Resps.html(request, "500--服务器出了点故障", httpConfig.getCharset());
			return ret;
		} finally {
			if (ret != null) {
				try {
					processCookieAfterHandler(request, requestLine, ret);
					if (httpServerInterceptor != null) {
						httpServerInterceptor.doAfterHandler(request, requestLine, ret);
					}
				} catch (Throwable e) {
					logError(request, requestLine, e);
				} finally {
					gzip(request, ret);
				}

				//				try {
				//					if (ret.isStaticRes() && (ret.getCookies() == null || ret.getCookies().size() == 0)) {
				//						ByteBuffer byteBuffer = HttpResponseEncoder.encode(ret, channelContext.getGroupContext(), channelContext, true);
				//						byte[] encodedBytes = byteBuffer.array();
				//						ret.setEncodedBytes(encodedBytes);

				//						GuavaCache guavaCache = GuavaCache.getCache(STATIC_RES_CACHENAME);
				//						guavaCache.put(requestLine.getPath(), ret);
				//					}
				//				} catch (Throwable e) {
				//					logError(request, requestLine, e);
				//				}
			}
		}
	}

	private void logError(HttpRequest request, RequestLine requestLine, Throwable e) {
		StringBuilder sb = new StringBuilder();
		sb.append("\r\n").append("remote  :").append(request.getRemote());
		sb.append("\r\n").append("request :").append(requestLine.getLine());
		log.error(sb.toString(), e);

	}

	private void processCookieAfterHandler(HttpRequest request, RequestLine requestLine, HttpResponse httpResponse) throws ExecutionException {
		HttpSession httpSession = request.getHttpSession();//(HttpSession) channelContext.getAttribute();//.getHttpSession();//not null
		Cookie cookie = getSessionCookie(request, httpConfig);
		String sessionId = null;

		if (cookie == null) {
			createSessionCookie(request, httpSession, httpResponse);
			log.info("{} 创建会话Cookie, {}", request.getChannelContext(), cookie);
		} else {
			sessionId = cookie.getValue();
			HttpSession httpSession1 = (HttpSession) httpConfig.getSessionStore().get(sessionId);

			if (httpSession1 == null) {//有cookie但是超时了
				createSessionCookie(request, httpSession, httpResponse);
			}
		}
	}

	/**
	 * 
	 * @param request
	 * @param httpSession
	 * @param httpResponse
	 * @return
	 * @author tanyaowu
	 */
	private Cookie createSessionCookie(HttpRequest request, HttpSession httpSession, HttpResponse httpResponse) {
		String sessionId = httpSession.getId();
		//		String host = request.getHost();
		String domain = request.getDomain();

		String name = httpConfig.getSessionCookieName();
		long maxAge = httpConfig.getSessionTimeout();
		//				maxAge = Long.MAX_VALUE; //把过期时间掌握在服务器端

		Cookie sessionCookie = new Cookie(domain, name, sessionId, maxAge);

		if (sessionCookieDecorator != null) {
			sessionCookieDecorator.decorate(sessionCookie);
		}
		httpResponse.addCookie(sessionCookie);

		httpConfig.getSessionStore().put(sessionId, httpSession);

		return sessionCookie;
	}

	private void processCookieBeforeHandler(HttpRequest request, RequestLine requestLine) throws ExecutionException {
		Cookie cookie = getSessionCookie(request, httpConfig);
		HttpSession httpSession = null;
		if (cookie == null) {
			httpSession = createSession(request);
		} else {
			//			httpSession = (HttpSession)httpSession.getAttribute(SESSIONID_KEY);//loadingCache.getIfPresent(sessionCookie.getValue());
			String sessionId = cookie.getValue();
			httpSession = (HttpSession) httpConfig.getSessionStore().get(sessionId);
			if (httpSession == null) {
				log.info("{} session【{}】超时", request.getChannelContext(), sessionId);
				httpSession = createSession(request);
			}
		}
		request.setHttpSession(httpSession);
	}

	@Override
	public HttpResponse resp404(HttpRequest request, RequestLine requestLine) {
		return Resps.resp404(request, requestLine, httpConfig);
	}

	@Override
	public HttpResponse resp500(HttpRequest request, RequestLine requestLine, Throwable throwable) {
		if (throwableHandler != null) {
			return throwableHandler.handler(request, requestLine, throwable);
		}
		return Resps.resp500(request, requestLine, httpConfig, throwable);
	}

	/**
	 * @param httpConfig the httpConfig to set
	 */
	public void setHttpConfig(HttpConfig httpConfig) {
		this.httpConfig = httpConfig;
	}

	public void setHttpServerInterceptor(HttpServerInterceptor httpServerInterceptor) {
		this.httpServerInterceptor = httpServerInterceptor;
	}

	/**
	 * @param staticResCache the staticResCache to set
	 */
	public void setStaticResCache(GuavaCache staticResCache) {
		this.staticResCache = staticResCache;
	}

	@Override
	public void clearStaticResCache(HttpRequest request) {
		if (staticResCache != null) {
			staticResCache.clear();
		}
	}

	public HttpSessionListener getHttpSessionListener() {
		return httpSessionListener;
	}

	public void setHttpSessionListener(HttpSessionListener httpSessionListener) {
		this.httpSessionListener = httpSessionListener;
	}

	public SessionCookieDecorator getSessionCookieDecorator() {
		return sessionCookieDecorator;
	}

	public void setSessionCookieDecorator(SessionCookieDecorator sessionCookieDecorator) {
		this.sessionCookieDecorator = sessionCookieDecorator;
	}

	public IpPathAccessStats getIpPathAccessStats() {
		return ipPathAccessStats;
	}

	public void setIpPathAccessStats(IpPathAccessStats ipPathAccessStats) {
		this.ipPathAccessStats = ipPathAccessStats;
	}

	public FreemarkerConfig getFreemarkerConfig() {
		return freemarkerConfig;
	}

	public void setFreemarkerConfig(FreemarkerConfig freemarkerConfig) {
		this.freemarkerConfig = freemarkerConfig;
	}

	public ThrowableHandler getThrowableHandler() {
		return throwableHandler;
	}

	public void setThrowableHandler(ThrowableHandler throwableHandler) {
		this.throwableHandler = throwableHandler;
	}
}
