/*
 * Copyright 2002-2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.test.context.junit4.statements;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.junit.runners.model.MultipleFailureException;
import org.junit.runners.model.Statement;

import org.springframework.test.context.TestContextManager;

/**
 * {@code RunAfterTestExecutionCallbacks} is a custom JUnit {@link Statement}
 * which allows the <em>Spring TestContext Framework</em> to be plugged into the
 * JUnit 4 execution chain by calling {@link TestContextManager#afterTestExecution
 * afterTestExecution()} on the supplied {@link TestContextManager}.
 *
 * <p><strong>NOTE:</strong> This class requires JUnit 4.9 or higher.
 *
 * @author Sam Brannen
 * @since 5.0
 * @see #evaluate()
 * @see RunBeforeTestExecutionCallbacks
 */
public class RunAfterTestExecutionCallbacks extends Statement {

	private final Statement next;

	private final Object testInstance;

	private final Method testMethod;

	private final TestContextManager testContextManager;


	/**
	 * Construct a new {@code RunAfterTestExecutionCallbacks} statement.
	 * @param next the next {@code Statement} in the execution chain
	 * @param testInstance the current test instance (never {@code null})
	 * @param testMethod the test method which has just been executed on the
	 * test instance
	 * @param testContextManager the TestContextManager upon which to call
	 * {@code afterTestExecution()}
	 */
	public RunAfterTestExecutionCallbacks(Statement next, Object testInstance, Method testMethod,
			TestContextManager testContextManager) {

		this.next = next;
		this.testInstance = testInstance;
		this.testMethod = testMethod;
		this.testContextManager = testContextManager;
	}

	/**
	 * Evaluate the next {@link Statement} in the execution chain (typically an
	 * instance of {@link RunBeforeTestExecutionCallbacks}), catching any exceptions
	 * thrown, and then invoke {@link TestContextManager#afterTestExecution} supplying
	 * the first caught exception (if any).
	 * <p>If the invocation of {@code afterTestExecution()} throws an exception, that
	 * exception will also be tracked. Multiple exceptions will be combined into a
	 * {@link MultipleFailureException}.
	 */
	@Override
	public void evaluate() throws Throwable {
		Throwable testException = null;
		List<Throwable> errors = new ArrayList<>();
		try {
			this.next.evaluate();
		}
		catch (Throwable ex) {
			testException = ex;
			errors.add(ex);
		}

		try {
			this.testContextManager.afterTestExecution(this.testInstance, this.testMethod, testException);
		}
		catch (Throwable ex) {
			errors.add(ex);
		}

		MultipleFailureException.assertEmpty(errors);
	}

}
