/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.home.cache;

import java.io.File;
import java.io.IOException;
import java.util.Random;
import javax.annotation.CheckForNull;
import org.apache.commons.io.FileUtils;
import org.sonar.home.cache.FileHashes;
import org.sonar.home.log.Log;

public class FileCache {
    private static final int TEMP_FILE_ATTEMPTS = 1000;
    private final File dir;
    private final File tmpDir;
    private final FileHashes hashes;
    private final Log log;

    FileCache(File dir, Log log, FileHashes fileHashes) {
        this.hashes = fileHashes;
        this.log = log;
        this.dir = this.createDir(dir, log, "user cache");
        log.info(String.format("User cache: %s", dir.getAbsolutePath()));
        this.tmpDir = this.createDir(new File(dir, "_tmp"), log, "temp dir");
    }

    public static FileCache create(File dir, Log log) {
        return new FileCache(dir, log, new FileHashes());
    }

    public File getDir() {
        return this.dir;
    }

    @CheckForNull
    public File get(String filename, String hash) {
        File cachedFile = new File(new File(this.dir, hash), filename);
        if (cachedFile.exists()) {
            return cachedFile;
        }
        this.log.debug(String.format("No file found in the cache with name %s and hash %s", filename, hash));
        return null;
    }

    public File get(String filename, String hash, Downloader downloader) {
        File hashDir = this.hashDir(hash);
        File targetFile = new File(hashDir, filename);
        if (!targetFile.exists()) {
            File tempFile = this.newTempFile();
            this.download(downloader, filename, tempFile);
            String downloadedHash = this.hashes.of(tempFile);
            if (!hash.equals(downloadedHash)) {
                throw new IllegalStateException("INVALID HASH: File " + tempFile.getAbsolutePath() + " was expected to have hash " + hash + " but was downloaded with hash " + downloadedHash);
            }
            this.mkdirQuietly(hashDir);
            this.renameQuietly(tempFile, targetFile);
        }
        return targetFile;
    }

    private void download(Downloader downloader, String filename, File tempFile) {
        try {
            downloader.download(filename, tempFile);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to download " + filename + " to " + tempFile, e);
        }
    }

    private void renameQuietly(File sourceFile, File targetFile) {
        boolean rename = sourceFile.renameTo(targetFile);
        if (!rename && !targetFile.exists()) {
            this.log.warn(String.format("Unable to rename %s to %s", sourceFile.getAbsolutePath(), targetFile.getAbsolutePath()));
            this.log.warn(String.format("A copy/delete will be tempted but with no garantee of atomicity", new Object[0]));
            try {
                FileUtils.moveFile((File)sourceFile, (File)targetFile);
            }
            catch (IOException e) {
                throw new IllegalStateException("Fail to move " + sourceFile.getAbsolutePath() + " to " + targetFile, e);
            }
        }
    }

    private File hashDir(String hash) {
        return new File(this.dir, hash);
    }

    private void mkdirQuietly(File hashDir) {
        try {
            FileUtils.forceMkdir((File)hashDir);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create cache directory: " + hashDir, e);
        }
    }

    private File newTempFile() {
        String baseName = System.currentTimeMillis() + "-";
        Random random = new Random();
        for (int counter = 0; counter < 1000; ++counter) {
            try {
                String filename = baseName + random.nextInt(1000);
                File tempFile = new File(this.tmpDir, filename);
                if (!tempFile.createNewFile()) continue;
                return tempFile;
            }
            catch (IOException e) {
                if (counter != 999) continue;
                throw new IllegalStateException();
            }
        }
        throw new IllegalStateException("Fail to create temporary file in " + this.tmpDir);
    }

    private File createDir(File dir, Log log, String debugTitle) {
        if (!dir.isDirectory() || !dir.exists()) {
            log.debug("Create : " + dir.getAbsolutePath());
            try {
                FileUtils.forceMkdir((File)dir);
            }
            catch (IOException e) {
                throw new IllegalStateException("Unable to create " + debugTitle + dir.getAbsolutePath(), e);
            }
        }
        return dir;
    }

    public static interface Downloader {
        public void download(String var1, File var2) throws IOException;
    }
}

