/*
 * Decompiled with CFR 0.152.
 */
package com.subgraph.orchid.directory;

import com.subgraph.orchid.Document;
import com.subgraph.orchid.TorConfig;
import com.subgraph.orchid.crypto.TorRandom;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.WritableByteChannel;
import java.util.List;
import java.util.logging.Logger;

public class DirectoryStoreFile {
    private static final Logger logger = Logger.getLogger(DirectoryStoreFile.class.getName());
    private static final ByteBuffer EMPTY_BUFFER = ByteBuffer.allocate(0);
    private static final TorRandom random = new TorRandom();
    private final TorConfig config;
    private final String cacheFilename;
    private RandomAccessFile openFile;
    private boolean openFileFailed;
    private boolean directoryCreationFailed;

    DirectoryStoreFile(TorConfig config, String cacheFilename) {
        this.config = config;
        this.cacheFilename = cacheFilename;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeData(ByteBuffer data) {
        File tempFile = this.createTempFile();
        FileOutputStream fos = this.openFileOutputStream(tempFile);
        if (fos == null) {
            return;
        }
        try {
            this.writeAllToChannel(fos.getChannel(), data);
            this.quietClose(fos);
            this.installTempFile(tempFile);
        }
        catch (IOException e) {
            logger.warning("I/O error writing to temporary cache file " + tempFile + " : " + e);
            return;
        }
        finally {
            this.quietClose(fos);
            tempFile.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeDocuments(List<? extends Document> documents) {
        File tempFile = this.createTempFile();
        FileOutputStream fos = this.openFileOutputStream(tempFile);
        if (fos == null) {
            return;
        }
        try {
            this.writeDocumentsToChannel(fos.getChannel(), documents);
            this.quietClose(fos);
            this.installTempFile(tempFile);
        }
        catch (IOException e) {
            logger.warning("I/O error writing to temporary cache file " + tempFile + " : " + e);
            return;
        }
        finally {
            this.quietClose(fos);
            tempFile.delete();
        }
    }

    private FileOutputStream openFileOutputStream(File file) {
        try {
            this.createDirectoryIfMissing();
            return new FileOutputStream(file);
        }
        catch (FileNotFoundException e) {
            logger.warning("Failed to open file " + file + " : " + e);
            return null;
        }
    }

    public void appendDocuments(List<? extends Document> documents) {
        if (!this.ensureOpened()) {
            return;
        }
        try {
            FileChannel channel = this.openFile.getChannel();
            channel.position(channel.size());
            this.writeDocumentsToChannel(channel, documents);
            channel.force(true);
        }
        catch (IOException e) {
            logger.warning("I/O error writing to cache file " + this.cacheFilename);
            return;
        }
    }

    public ByteBuffer loadContents() {
        if (!this.fileExists() || !this.ensureOpened()) {
            return EMPTY_BUFFER;
        }
        try {
            return this.readAllFromChannel(this.openFile.getChannel());
        }
        catch (IOException e) {
            logger.warning("I/O error reading cache file " + this.cacheFilename + " : " + e);
            return EMPTY_BUFFER;
        }
    }

    private ByteBuffer readAllFromChannel(FileChannel channel) throws IOException {
        channel.position(0L);
        ByteBuffer buffer = this.createBufferForChannel(channel);
        while (buffer.hasRemaining()) {
            if (channel.read(buffer) != -1) continue;
            logger.warning("Unexpected EOF reading from cache file");
            return EMPTY_BUFFER;
        }
        buffer.rewind();
        return buffer;
    }

    private ByteBuffer createBufferForChannel(FileChannel channel) throws IOException {
        int sz = (int)(channel.size() & 0xFFFFFFFFFFFFFFFFL);
        return ByteBuffer.allocateDirect(sz);
    }

    void close() {
        if (this.openFile != null) {
            this.quietClose(this.openFile);
            this.openFile = null;
        }
    }

    private boolean fileExists() {
        File file = this.getFile();
        return file.exists();
    }

    private boolean ensureOpened() {
        if (this.openFileFailed) {
            return false;
        }
        if (this.openFile != null) {
            return true;
        }
        this.openFile = this.openFile();
        return this.openFile != null;
    }

    private RandomAccessFile openFile() {
        try {
            File f = new File(this.config.getDataDirectory(), this.cacheFilename);
            this.createDirectoryIfMissing();
            return new RandomAccessFile(f, "rw");
        }
        catch (FileNotFoundException e) {
            this.openFileFailed = true;
            logger.warning("Failed to open cache file " + this.cacheFilename);
            return null;
        }
    }

    private void installTempFile(File tempFile) {
        this.close();
        File target = this.getFile();
        if (target.exists() && !target.delete()) {
            logger.warning("Failed to delete file " + target);
        }
        if (!tempFile.renameTo(target)) {
            logger.warning("Failed to rename temp file " + tempFile + " to " + target);
        }
        tempFile.delete();
        this.ensureOpened();
    }

    private File createTempFile() {
        long n = random.nextLong();
        File f = new File(this.config.getDataDirectory(), this.cacheFilename + Long.toString(n));
        f.deleteOnExit();
        return f;
    }

    private void writeDocumentsToChannel(FileChannel channel, List<? extends Document> documents) throws IOException {
        for (Document document : documents) {
            this.writeAllToChannel(channel, document.getRawDocumentBytes());
        }
    }

    private void writeAllToChannel(WritableByteChannel channel, ByteBuffer data) throws IOException {
        data.rewind();
        while (data.hasRemaining()) {
            channel.write(data);
        }
    }

    private void quietClose(Closeable closeable) {
        try {
            closeable.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private File getFile() {
        return new File(this.config.getDataDirectory(), this.cacheFilename);
    }

    public void remove() {
        this.close();
        this.getFile().delete();
    }

    private void createDirectoryIfMissing() {
        if (this.directoryCreationFailed) {
            return;
        }
        File dd = this.config.getDataDirectory();
        if (!dd.exists() && !dd.mkdirs()) {
            this.directoryCreationFailed = true;
            logger.warning("Failed to create data directory " + dd);
        }
    }
}

