/*
 * Decompiled with CFR 0.152.
 */
package com.aliyun.datahub.client.http.converter.batch;

import com.aliyun.datahub.client.exception.DatahubClientException;
import com.aliyun.datahub.client.exception.InvalidParameterException;
import com.aliyun.datahub.client.exception.MalformedRecordException;
import com.aliyun.datahub.client.http.converter.batch.BatchUtil;
import com.aliyun.datahub.client.model.Field;
import com.aliyun.datahub.client.model.RecordSchema;
import com.aliyun.datahub.shaded.org.apache.commons.codec.Charsets;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public class BinaryRecord {
    private static final int RECORD_HEADER_SIZE = 16;
    private static final int BYTE_SIZE_ONE_FIELD = 8;
    private static final int FIELD_COUNT_BYTE_SIZE = 4;
    private static final int INT_BYTE_SIZE = 4;
    private static final byte[] PADDING_BYTES = new byte[]{0, 0, 0, 0, 0, 0, 0, 0};
    private int fieldCnt;
    private int fieldPos;
    private int nextPos;
    private int attrLength;
    private RecordSchema schema;
    private byte[] recordBuffer;
    private Map<String, String> attrMap = new HashMap<String, String>();
    private boolean hasInitAttrMap;
    private RecordHeader recordHeader;

    private BinaryRecord(byte[] buffer, RecordHeader header, RecordSchema schema) {
        this.hasInitAttrMap = false;
        this.recordBuffer = buffer;
        this.schema = schema;
        this.fieldCnt = schema != null ? schema.getFields().size() : 1;
        this.fieldPos = this.getFixHeaderLength(this.fieldCnt);
        this.recordHeader = header;
    }

    public BinaryRecord(RecordSchema schema) {
        this.hasInitAttrMap = true;
        this.schema = schema;
        this.fieldCnt = schema != null ? schema.getFields().size() : 1;
        int minAllocSize = this.getMinAllocSize();
        this.recordBuffer = new byte[minAllocSize];
        this.fieldPos = this.getFixHeaderLength(this.fieldCnt);
        this.nextPos = minAllocSize;
    }

    public Object getField(int pos) {
        if (this.isFieldNull(pos)) {
            return null;
        }
        try {
            if (this.schema == null) {
                return this.readStrField(0).getBytes(Charsets.UTF_8);
            }
            Field field = this.schema.getField(pos);
            switch (field.getType()) {
                case STRING: {
                    return this.readStrField(pos);
                }
                case DECIMAL: {
                    return new BigDecimal(this.readStrField(pos));
                }
                case FLOAT: {
                    return Float.valueOf(this.readField(pos).getFloat());
                }
                case DOUBLE: {
                    return this.readField(pos).getDouble();
                }
                case BOOLEAN: {
                    return this.readField(pos).getLong() != 0L;
                }
                case INTEGER: {
                    return this.readField(pos).getInt();
                }
                case TINYINT: {
                    return (byte)this.readField(pos).getInt();
                }
                case SMALLINT: {
                    return this.readField(pos).getShort();
                }
                case BIGINT: 
                case TIMESTAMP: {
                    return this.readField(pos).getLong();
                }
            }
            throw new InvalidParameterException("Unknown schema type");
        }
        catch (Exception e) {
            throw new MalformedRecordException("Parse field fail. position:" + pos + ", error:" + e.getMessage());
        }
    }

    public void addAttribute(String key, String value) {
        this.attrMap.put(key, value);
        this.attrLength += 8 + key.length() + value.length();
    }

    public Map<String, String> getAttrMap() {
        this.initAttrMapIfNeed();
        return this.attrMap;
    }

    private void initAttrMapIfNeed() {
        if (this.hasInitAttrMap) {
            return;
        }
        RecordHeader recordHeader = this.constructRecordHeader();
        int offset = recordHeader.attrOffset;
        int attrSize = BatchUtil.readInt(this.recordBuffer, offset);
        if (attrSize != 0 && this.attrMap == null) {
            this.attrMap = new HashMap<String, String>();
        }
        offset += 4;
        for (int i = 0; i < attrSize; ++i) {
            int len = BatchUtil.readInt(this.recordBuffer, offset);
            String keyStr = new String(this.recordBuffer, offset += 4, len);
            offset += len;
            len = BatchUtil.readInt(this.recordBuffer, offset);
            String valueStr = new String(this.recordBuffer, offset += 4, len);
            this.attrMap.put(keyStr, valueStr);
            offset += len;
        }
    }

    public int getRecordSize() {
        return 4 + this.attrLength + this.nextPos;
    }

    private String readStrField(int pos) {
        boolean isLittleStr;
        ByteBuffer buffer = this.readField(pos);
        long data = buffer.getLong();
        boolean bl = isLittleStr = (data & Long.MIN_VALUE) != 0L;
        if (isLittleStr) {
            int len = (int)(data >> 56 & 7L);
            byte[] bytes = BatchUtil.parseLong(data);
            return new String(bytes, 0, len);
        }
        int strOffset = 16 + (int)(data >> 32);
        return new String(this.recordBuffer, strOffset, (int)data);
    }

    private ByteBuffer readField(int pos) {
        int offset = this.getFieldOffset(pos);
        return ByteBuffer.wrap(this.recordBuffer, offset, 8).order(ByteOrder.LITTLE_ENDIAN);
    }

    public void setField(int pos, Object value) {
        this.setNotNullAt(pos);
        if (this.schema == null) {
            if (!(value instanceof byte[])) {
                throw new DatahubClientException("Only support write byte[] for no schema");
            }
            this.writeStrField(0, (byte[])value);
            return;
        }
        Field field = this.schema.getField(pos);
        switch (field.getType()) {
            case STRING: {
                this.writeStrField(pos, ((String)value).getBytes(Charsets.UTF_8));
                break;
            }
            case DECIMAL: {
                this.writeStrField(pos, ((BigDecimal)value).toPlainString().getBytes(Charsets.UTF_8));
                break;
            }
            case FLOAT: {
                this.writeField(pos, BatchUtil.parseFloat(((Float)value).floatValue()));
                break;
            }
            case DOUBLE: {
                this.writeField(pos, BatchUtil.parseDouble((Double)value));
                break;
            }
            case BOOLEAN: {
                long data = (Boolean)value != false ? 1L : 0L;
                this.writeField(pos, BatchUtil.parseLong(data));
                break;
            }
            default: {
                this.writeField(pos, BatchUtil.parseLong(new Long(value.toString())));
            }
        }
    }

    private void setNotNullAt(int pos) {
        byte value;
        this.checkPosValid(pos);
        int nullOffset = 20 + (pos >> 3);
        this.recordBuffer[nullOffset] = value = (byte)(this.recordBuffer[nullOffset] | 1 << (pos & 7));
    }

    private boolean isFieldNull(int pos) {
        this.checkPosValid(pos);
        int nullOffset = 20 + (pos >> 3);
        byte value = (byte)(this.recordBuffer[nullOffset] & 1 << (pos & 7));
        return value == 0;
    }

    private void writeStrField(int pos, byte[] bytes) {
        if (bytes.length <= 7) {
            this.writeField(pos, BatchUtil.parseLittleStr(bytes));
        } else {
            this.writeBigStr(pos, bytes);
        }
    }

    private void writeBigStr(int pos, byte[] bytes) {
        int byteLength = bytes.length;
        int needSize = this.alignSize(byteLength);
        this.ensureBufferCapability(needSize);
        long offsetAndSize = (long)(this.nextPos - 16) << 32 | (long)byteLength;
        this.writeField(pos, BatchUtil.parseLong(offsetAndSize));
        System.arraycopy(bytes, 0, this.recordBuffer, this.nextPos, bytes.length);
        int left = needSize - byteLength;
        if (left > 0) {
            System.arraycopy(PADDING_BYTES, 0, this.recordBuffer, this.nextPos + byteLength, left);
        }
        this.nextPos += needSize;
    }

    private void ensureBufferCapability(int needSize) {
        int minCap = this.nextPos + needSize;
        int currCap = this.recordBuffer.length;
        if (minCap > currCap) {
            int newCap = currCap * 2;
            if (newCap < minCap) {
                newCap = minCap;
            }
            this.recordBuffer = Arrays.copyOf(this.recordBuffer, newCap);
        }
    }

    private void writeField(int pos, byte[] bytes) {
        int offset = this.getFieldOffset(pos);
        System.arraycopy(bytes, 0, this.recordBuffer, offset, bytes.length);
    }

    private int alignSize(int size) {
        return size + 7 & 0xFFFFFFF8;
    }

    private int getFieldOffset(int pos) {
        return this.fieldPos + pos * 8;
    }

    private int getMinAllocSize() {
        return this.getFixHeaderLength(this.fieldCnt) + this.fieldCnt * 8;
    }

    private int getFixHeaderLength(int fieldCount) {
        return 20 + (fieldCount + 63 >> 6 << 3);
    }

    private RecordHeader constructRecordHeader() {
        if (this.recordHeader == null) {
            this.recordHeader = new RecordHeader(){
                {
                    this.setEncodeType(0);
                    this.setSchemaVersion(0);
                    this.setTotalSize(BinaryRecord.this.getRecordSize());
                    this.setAttrOffset(BinaryRecord.this.nextPos);
                }
            };
        }
        return this.recordHeader;
    }

    public void serialize(ByteArrayOutputStream output) throws IOException {
        RecordHeader header = this.constructRecordHeader();
        byte[] headerBuffer = RecordHeader.serialize(header);
        System.arraycopy(headerBuffer, 0, this.recordBuffer, 0, headerBuffer.length);
        output.write(this.recordBuffer, 0, this.nextPos);
        output.write(BatchUtil.parseInt(this.attrMap.size()));
        for (Map.Entry<String, String> entry : this.attrMap.entrySet()) {
            output.write(BatchUtil.parseInt(entry.getKey().length()));
            output.write(entry.getKey().getBytes(Charsets.UTF_8));
            output.write(BatchUtil.parseInt(entry.getValue().length()));
            output.write(entry.getValue().getBytes(Charsets.UTF_8));
        }
    }

    public static BinaryRecord parseFrom(ByteArrayInputStream input, RecordSchema schema) {
        int left = input.available();
        input.mark(0);
        RecordHeader header = RecordHeader.parseFrom(input);
        if (left < header.getTotalSize()) {
            throw new DatahubClientException("Check record header length fail");
        }
        input.reset();
        byte[] bytes = new byte[header.getTotalSize()];
        int length = input.read(bytes, 0, header.totalSize);
        if (length != header.totalSize) {
            throw new DatahubClientException("Check record total size fail");
        }
        return new BinaryRecord(bytes, header, schema);
    }

    private void checkPosValid(int pos) {
        if (pos >= this.fieldCnt) {
            throw new DatahubClientException("Invalid position. position:" + pos + ", fieldCount:" + this.fieldCnt);
        }
    }

    private static class RecordHeader {
        private static final ThreadLocal<ByteBuffer> BYTE_BUFFER = ThreadLocal.withInitial(() -> ByteBuffer.allocate(16).order(ByteOrder.LITTLE_ENDIAN));
        private int encodeType;
        private int schemaVersion;
        private int totalSize;
        private int attrOffset;

        private RecordHeader() {
        }

        public int getEncodeType() {
            return this.encodeType;
        }

        public void setEncodeType(int encodeType) {
            this.encodeType = encodeType;
        }

        public int getSchemaVersion() {
            return this.schemaVersion;
        }

        public void setSchemaVersion(int schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        public int getTotalSize() {
            return this.totalSize;
        }

        public void setTotalSize(int totalSize) {
            this.totalSize = totalSize;
        }

        public int getAttrOffset() {
            return this.attrOffset;
        }

        public void setAttrOffset(int attrOffset) {
            this.attrOffset = attrOffset;
        }

        public static RecordHeader parseFrom(ByteArrayInputStream input) {
            final ByteBuffer byteBuffer = BYTE_BUFFER.get();
            byte[] buffer = new byte[16];
            int len = input.read(buffer, 0, 16);
            if (len < 16) {
                throw new DatahubClientException("read batch header fail");
            }
            byteBuffer.clear();
            byteBuffer.put(buffer);
            byteBuffer.flip();
            return new RecordHeader(){
                {
                    this.setEncodeType(byteBuffer.getInt());
                    this.setSchemaVersion(byteBuffer.getInt());
                    this.setTotalSize(byteBuffer.getInt());
                    this.setAttrOffset(byteBuffer.getInt());
                }
            };
        }

        public static byte[] serialize(RecordHeader header) {
            ByteBuffer byteBuffer = BYTE_BUFFER.get();
            byteBuffer.clear();
            byteBuffer.putInt(header.getEncodeType());
            byteBuffer.putInt(header.getSchemaVersion());
            byteBuffer.putInt(header.getTotalSize());
            byteBuffer.putInt(header.getAttrOffset());
            return byteBuffer.array();
        }
    }
}

