/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dubbo.security.cert;

import io.grpc.Channel;
import io.grpc.ClientInterceptor;
import io.grpc.Metadata;
import io.grpc.netty.shaded.io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import io.grpc.netty.shaded.io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import io.grpc.stub.MetadataUtils;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.ECGenParameterSpec;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.dubbo.auth.v1alpha1.DubboCertificateRequest;
import org.apache.dubbo.auth.v1alpha1.DubboCertificateResponse;
import org.apache.dubbo.auth.v1alpha1.DubboCertificateServiceGrpc;
import org.apache.dubbo.common.logger.ErrorTypeAwareLogger;
import org.apache.dubbo.common.logger.LoggerFactory;
import org.apache.dubbo.common.threadpool.manager.FrameworkExecutorRepository;
import org.apache.dubbo.common.utils.IOUtils;
import org.apache.dubbo.common.utils.StringUtils;
import org.apache.dubbo.rpc.model.FrameworkModel;
import org.apache.dubbo.security.cert.CertConfig;
import org.apache.dubbo.security.cert.CertPair;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;

public class DubboCertManager {
    private static final ErrorTypeAwareLogger logger = LoggerFactory.getErrorTypeAwareLogger(DubboCertManager.class);
    private final FrameworkModel frameworkModel;
    protected volatile Channel channel;
    protected volatile CertPair certPair;
    protected volatile CertConfig certConfig;
    protected volatile ScheduledFuture<?> refreshFuture;

    public DubboCertManager(FrameworkModel frameworkModel) {
        this.frameworkModel = frameworkModel;
    }

    public synchronized void connect(CertConfig certConfig) {
        if (this.channel != null) {
            logger.error("99-0", "", "", "Dubbo Cert Authority server is already connected.");
            return;
        }
        if (certConfig == null) {
            return;
        }
        if (StringUtils.isEmpty((String)certConfig.getRemoteAddress())) {
            return;
        }
        if (StringUtils.isNotEmpty((String)certConfig.getEnvType()) && !"Kubernetes".equalsIgnoreCase(certConfig.getEnvType())) {
            throw new IllegalArgumentException("Only support Kubernetes env now.");
        }
        this.connect0(certConfig);
        this.certConfig = certConfig;
        this.generateCert();
        this.scheduleRefresh();
    }

    protected void scheduleRefresh() {
        FrameworkExecutorRepository repository = (FrameworkExecutorRepository)this.frameworkModel.getBeanFactory().getBean(FrameworkExecutorRepository.class);
        this.refreshFuture = repository.getSharedScheduledExecutor().scheduleAtFixedRate(this::generateCert, this.certConfig.getRefreshInterval(), this.certConfig.getRefreshInterval(), TimeUnit.MILLISECONDS);
    }

    protected void connect0(CertConfig certConfig) {
        String caCertPath = certConfig.getCaCertPath();
        String remoteAddress = certConfig.getRemoteAddress();
        logger.info("Try to connect to Dubbo Cert Authority server: " + remoteAddress + ", caCertPath: " + remoteAddress);
        try {
            if (StringUtils.isNotEmpty((String)caCertPath)) {
                this.channel = NettyChannelBuilder.forTarget((String)remoteAddress).sslContext(GrpcSslContexts.forClient().trustManager(new File(caCertPath)).build()).build();
            } else {
                logger.warn("5-43", "", "", "No caCertPath is provided, will use insecure connection.");
                this.channel = NettyChannelBuilder.forTarget((String)remoteAddress).sslContext(GrpcSslContexts.forClient().trustManager(InsecureTrustManagerFactory.INSTANCE).build()).build();
            }
        }
        catch (Exception e) {
            logger.error("5-41", "", "", "Failed to load SSL cert file.", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    public synchronized void disConnect() {
        if (this.refreshFuture != null) {
            this.refreshFuture.cancel(true);
            this.refreshFuture = null;
        }
        if (this.channel != null) {
            this.channel = null;
        }
    }

    public boolean isConnected() {
        return this.certConfig != null && this.channel != null && this.certPair != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected CertPair generateCert() {
        if (this.certPair != null && !this.certPair.isExpire()) {
            return this.certPair;
        }
        DubboCertManager dubboCertManager = this;
        synchronized (dubboCertManager) {
            if (this.certPair == null || this.certPair.isExpire()) {
                try {
                    logger.info("Try to generate cert from Dubbo Certificate Authority.");
                    CertPair certFromRemote = this.refreshCert();
                    if (certFromRemote != null) {
                        this.certPair = certFromRemote;
                    } else {
                        logger.error("5-42", "", "", "Generate Cert from Dubbo Certificate Authority failed.");
                    }
                }
                catch (Exception e) {
                    logger.error("1-26", "", "", "Generate Cert from Istio failed.", (Throwable)e);
                }
            }
        }
        return this.certPair;
    }

    protected CertPair refreshCert() throws IOException {
        KeyPair keyPair = DubboCertManager.signWithEcdsa();
        if (keyPair == null) {
            keyPair = DubboCertManager.signWithRsa();
        }
        if (keyPair == null) {
            logger.error("5-42", "", "", "Generate Key failed. Please check if your system support.");
            return null;
        }
        String csr = this.generateCsr(keyPair);
        DubboCertificateServiceGrpc.DubboCertificateServiceBlockingStub stub = DubboCertificateServiceGrpc.newBlockingStub(this.channel);
        stub = this.setHeaderIfNeed(stub);
        String privateKeyPem = this.generatePrivatePemKey(keyPair);
        DubboCertificateResponse certificateResponse = stub.createCertificate(this.generateRequest(csr));
        if (certificateResponse == null || !certificateResponse.getSuccess()) {
            logger.error("5-42", "", "", "Failed to generate cert from Dubbo Certificate Authority. Message: " + (certificateResponse == null ? "null" : certificateResponse.getMessage()));
            return null;
        }
        logger.info("Successfully generate cert from Dubbo Certificate Authority. Cert expire time: " + certificateResponse.getExpireTime());
        return new CertPair(privateKeyPem, certificateResponse.getCertPem(), String.join((CharSequence)"\n", (Iterable<? extends CharSequence>)certificateResponse.getTrustCertsList()), certificateResponse.getExpireTime());
    }

    private DubboCertificateServiceGrpc.DubboCertificateServiceBlockingStub setHeaderIfNeed(DubboCertificateServiceGrpc.DubboCertificateServiceBlockingStub stub) throws IOException {
        String oidcTokenPath = this.certConfig.getOidcTokenPath();
        if (StringUtils.isNotEmpty((String)oidcTokenPath)) {
            Metadata header = new Metadata();
            Metadata.Key key = Metadata.Key.of((String)"authorization", (Metadata.AsciiMarshaller)Metadata.ASCII_STRING_MARSHALLER);
            header.put(key, (Object)("Bearer " + IOUtils.read((Reader)new FileReader(oidcTokenPath)).replace("\n", "").replace("\t", "").replace("\r", "").trim()));
            stub = (DubboCertificateServiceGrpc.DubboCertificateServiceBlockingStub)stub.withInterceptors(new ClientInterceptor[]{MetadataUtils.newAttachHeadersInterceptor((Metadata)header)});
            logger.info("Use oidc token from " + oidcTokenPath + " to connect to Dubbo Certificate Authority.");
        } else {
            logger.warn("5-43", "", "", "Use insecure connection to connect to Dubbo Certificate Authority. Reason: No oidc token is provided.");
        }
        return stub;
    }

    protected static KeyPair signWithRsa() {
        KeyPair keyPair = null;
        try {
            KeyPairGenerator kpGenerator = KeyPairGenerator.getInstance("RSA");
            kpGenerator.initialize(4096);
            java.security.KeyPair keypair = kpGenerator.generateKeyPair();
            PublicKey publicKey = keypair.getPublic();
            PrivateKey privateKey = keypair.getPrivate();
            ContentSigner signer = new JcaContentSignerBuilder("SHA256WithRSA").build(keypair.getPrivate());
            keyPair = new KeyPair(publicKey, privateKey, signer);
        }
        catch (NoSuchAlgorithmException | OperatorCreationException e) {
            logger.error("5-42", "", "", "Generate Key with SHA256WithRSA algorithm failed. Please check if your system support.", e);
        }
        return keyPair;
    }

    protected static KeyPair signWithEcdsa() {
        KeyPair keyPair = null;
        try {
            ECGenParameterSpec ecSpec = new ECGenParameterSpec("secp256r1");
            KeyPairGenerator g = KeyPairGenerator.getInstance("EC");
            g.initialize(ecSpec, new SecureRandom());
            java.security.KeyPair keypair = g.generateKeyPair();
            PublicKey publicKey = keypair.getPublic();
            PrivateKey privateKey = keypair.getPrivate();
            ContentSigner signer = new JcaContentSignerBuilder("SHA256withECDSA").build(privateKey);
            keyPair = new KeyPair(publicKey, privateKey, signer);
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException | OperatorCreationException e) {
            logger.error("5-42", "", "", "Generate Key with secp256r1 algorithm failed. Please check if your system support. Will attempt to generate with RSA2048.", e);
        }
        return keyPair;
    }

    private DubboCertificateRequest generateRequest(String csr) {
        return DubboCertificateRequest.newBuilder().setCsr(csr).setType("CONNECTION").build();
    }

    private String generatePrivatePemKey(KeyPair keyPair) throws IOException {
        String key = this.generatePemKey("RSA PRIVATE KEY", keyPair.getPrivateKey().getEncoded());
        if (logger.isDebugEnabled()) {
            logger.debug("Generated Private Key. \n" + key);
        }
        return key;
    }

    private String generatePemKey(String type, byte[] content) throws IOException {
        PemObject pemObject = new PemObject(type, content);
        StringWriter str = new StringWriter();
        JcaPEMWriter jcaPEMWriter = new JcaPEMWriter((Writer)str);
        jcaPEMWriter.writeObject((PemObjectGenerator)pemObject);
        jcaPEMWriter.close();
        str.close();
        return str.toString();
    }

    private String generateCsr(KeyPair keyPair) throws IOException {
        PKCS10CertificationRequest request = new JcaPKCS10CertificationRequestBuilder(new X500Name("O=cluster.domain"), keyPair.getPublicKey()).build(keyPair.getSigner());
        String csr = this.generatePemKey("CERTIFICATE REQUEST", request.getEncoded());
        if (logger.isDebugEnabled()) {
            logger.debug("CSR Request to Dubbo Certificate Authorization. \n" + csr);
        }
        return csr;
    }

    protected static class KeyPair {
        private final PublicKey publicKey;
        private final PrivateKey privateKey;
        private final ContentSigner signer;

        public KeyPair(PublicKey publicKey, PrivateKey privateKey, ContentSigner signer) {
            this.publicKey = publicKey;
            this.privateKey = privateKey;
            this.signer = signer;
        }

        public PublicKey getPublicKey() {
            return this.publicKey;
        }

        public PrivateKey getPrivateKey() {
            return this.privateKey;
        }

        public ContentSigner getSigner() {
            return this.signer;
        }
    }
}

