/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.server;

import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import org.apache.druid.client.DirectDruidClient;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.guava.Sequence;
import org.apache.druid.java.util.common.guava.SequenceWrapper;
import org.apache.druid.java.util.common.guava.Sequences;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.java.util.emitter.service.ServiceEmitter;
import org.apache.druid.query.DefaultQueryConfig;
import org.apache.druid.query.DruidMetrics;
import org.apache.druid.query.GenericQueryMetricsFactory;
import org.apache.druid.query.Query;
import org.apache.druid.query.QueryContext;
import org.apache.druid.query.QueryContexts;
import org.apache.druid.query.QueryInterruptedException;
import org.apache.druid.query.QueryMetrics;
import org.apache.druid.query.QueryPlus;
import org.apache.druid.query.QuerySegmentWalker;
import org.apache.druid.query.QueryTimeoutException;
import org.apache.druid.query.QueryToolChest;
import org.apache.druid.query.QueryToolChestWarehouse;
import org.apache.druid.query.context.ConcurrentResponseContext;
import org.apache.druid.query.context.ResponseContext;
import org.apache.druid.server.QueryResource;
import org.apache.druid.server.QueryStats;
import org.apache.druid.server.RequestLogLine;
import org.apache.druid.server.log.RequestLogger;
import org.apache.druid.server.security.Access;
import org.apache.druid.server.security.Action;
import org.apache.druid.server.security.AuthConfig;
import org.apache.druid.server.security.AuthenticationResult;
import org.apache.druid.server.security.AuthorizationUtils;
import org.apache.druid.server.security.AuthorizerMapper;
import org.apache.druid.server.security.Resource;
import org.apache.druid.server.security.ResourceAction;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;

public class QueryLifecycle {
    private static final Logger log = new Logger(QueryLifecycle.class);
    private final QueryToolChestWarehouse warehouse;
    private final QuerySegmentWalker texasRanger;
    private final GenericQueryMetricsFactory queryMetricsFactory;
    private final ServiceEmitter emitter;
    private final RequestLogger requestLogger;
    private final AuthorizerMapper authorizerMapper;
    private final DefaultQueryConfig defaultQueryConfig;
    private final AuthConfig authConfig;
    private final long startMs;
    private final long startNs;
    private State state = State.NEW;
    private AuthenticationResult authenticationResult;
    private QueryToolChest toolChest;
    private @MonotonicNonNull Query<?> baseQuery;

    public QueryLifecycle(QueryToolChestWarehouse warehouse, QuerySegmentWalker texasRanger, GenericQueryMetricsFactory queryMetricsFactory, ServiceEmitter emitter, RequestLogger requestLogger, AuthorizerMapper authorizerMapper, DefaultQueryConfig defaultQueryConfig, AuthConfig authConfig, long startMs, long startNs) {
        this.warehouse = warehouse;
        this.texasRanger = texasRanger;
        this.queryMetricsFactory = queryMetricsFactory;
        this.emitter = emitter;
        this.requestLogger = requestLogger;
        this.authorizerMapper = authorizerMapper;
        this.defaultQueryConfig = defaultQueryConfig;
        this.authConfig = authConfig;
        this.startMs = startMs;
        this.startNs = startNs;
    }

    public <T> Sequence<T> runSimple(Query<T> query, AuthenticationResult authenticationResult, Access authorizationResult) {
        Sequence results;
        this.initialize(query);
        try {
            this.preAuthorized(authenticationResult, authorizationResult);
            if (!authorizationResult.isAllowed()) {
                throw new ISE("Unauthorized", new Object[0]);
            }
            QueryResponse queryResponse = this.execute();
            results = queryResponse.getResults();
        }
        catch (Throwable e) {
            this.emitLogsAndMetrics(e, null, -1L);
            throw e;
        }
        return Sequences.wrap((Sequence)results, (SequenceWrapper)new SequenceWrapper(){

            public void after(boolean isDone, Throwable thrown) {
                QueryLifecycle.this.emitLogsAndMetrics(thrown, null, -1L);
            }
        });
    }

    public void initialize(Query baseQuery) {
        this.transition(State.NEW, State.INITIALIZED);
        if (baseQuery.getQueryContext() == null) {
            QueryContext context = new QueryContext(baseQuery.getContext());
            context.addDefaultParam("queryId", (Object)UUID.randomUUID().toString());
            context.addDefaultParams(this.defaultQueryConfig.getContext());
            this.baseQuery = baseQuery.withOverriddenContext(context.getMergedParams());
        } else {
            baseQuery.getQueryContext().addDefaultParam("queryId", (Object)UUID.randomUUID().toString());
            baseQuery.getQueryContext().addDefaultParams(this.defaultQueryConfig.getContext());
            this.baseQuery = baseQuery;
        }
        this.toolChest = this.warehouse.getToolChest(this.baseQuery);
    }

    public Access authorize(HttpServletRequest req) {
        this.transition(State.INITIALIZED, State.AUTHORIZING);
        Set contextKeys = this.baseQuery.getQueryContext() == null ? this.baseQuery.getContext().keySet() : this.baseQuery.getQueryContext().getUserParams().keySet();
        Iterable resourcesToAuthorize = Iterables.concat((Iterable)Iterables.transform((Iterable)this.baseQuery.getDataSource().getTableNames(), AuthorizationUtils.DATASOURCE_READ_RA_GENERATOR), this.authConfig.authorizeQueryContextParams() ? Iterables.transform(contextKeys, contextParam -> new ResourceAction(new Resource((String)contextParam, "QUERY_CONTEXT"), Action.WRITE)) : Collections.emptyList());
        return this.doAuthorize(AuthorizationUtils.authenticationResultFromRequest(req), AuthorizationUtils.authorizeAllResourceActions(req, (Iterable<ResourceAction>)resourcesToAuthorize, this.authorizerMapper));
    }

    private void preAuthorized(AuthenticationResult authenticationResult, Access access) {
        this.transition(State.INITIALIZED, State.AUTHORIZING);
        this.doAuthorize(authenticationResult, access);
    }

    private Access doAuthorize(AuthenticationResult authenticationResult, Access authorizationResult) {
        Preconditions.checkNotNull((Object)authenticationResult, (Object)"authenticationResult");
        Preconditions.checkNotNull((Object)authorizationResult, (Object)"authorizationResult");
        if (!authorizationResult.isAllowed()) {
            this.transition(State.AUTHORIZING, State.UNAUTHORIZED);
        } else {
            this.transition(State.AUTHORIZING, State.AUTHORIZED);
        }
        this.authenticationResult = authenticationResult;
        return authorizationResult;
    }

    public QueryResponse execute() {
        this.transition(State.AUTHORIZED, State.EXECUTING);
        ConcurrentResponseContext responseContext = DirectDruidClient.makeResponseContextForQuery();
        Sequence res = QueryPlus.wrap(this.baseQuery).withIdentity(this.authenticationResult.getIdentity()).run(this.texasRanger, (ResponseContext)responseContext);
        return new QueryResponse(res == null ? Sequences.empty() : res, (ResponseContext)responseContext);
    }

    public void emitLogsAndMetrics(@Nullable Throwable e, @Nullable String remoteAddress, long bytesWritten) {
        if (this.baseQuery == null) {
            return;
        }
        if (this.state == State.DONE) {
            log.warn("Tried to emit logs and metrics twice for query[%s]!", new Object[]{this.baseQuery.getId()});
        }
        this.state = State.DONE;
        boolean success = e == null;
        try {
            long queryTimeNs = System.nanoTime() - this.startNs;
            QueryMetrics queryMetrics = DruidMetrics.makeRequestMetrics((GenericQueryMetricsFactory)this.queryMetricsFactory, (QueryToolChest)this.toolChest, this.baseQuery, (String)StringUtils.nullToEmptyNonDruidDataString((String)remoteAddress));
            queryMetrics.success(success);
            queryMetrics.reportQueryTime(queryTimeNs);
            if (bytesWritten >= 0L) {
                queryMetrics.reportQueryBytes(bytesWritten);
            }
            if (this.authenticationResult != null) {
                queryMetrics.identity(this.authenticationResult.getIdentity());
            }
            queryMetrics.emit(this.emitter);
            LinkedHashMap<String, Object> statsMap = new LinkedHashMap<String, Object>();
            statsMap.put("query/time", TimeUnit.NANOSECONDS.toMillis(queryTimeNs));
            statsMap.put("query/bytes", bytesWritten);
            statsMap.put("success", success);
            if (this.authenticationResult != null) {
                statsMap.put("identity", this.authenticationResult.getIdentity());
            }
            if (e != null) {
                statsMap.put("exception", e.toString());
                if (QueryContexts.isDebug(this.baseQuery)) {
                    log.warn(e, "Exception while processing queryId [%s]", new Object[]{this.baseQuery.getId()});
                } else {
                    log.noStackTrace().warn(e, "Exception while processing queryId [%s]", new Object[]{this.baseQuery.getId()});
                }
                if (e instanceof QueryInterruptedException || e instanceof QueryTimeoutException) {
                    statsMap.put("interrupted", true);
                    statsMap.put("reason", e.toString());
                }
            }
            this.requestLogger.logNativeQuery(RequestLogLine.forNative(this.baseQuery, DateTimes.utc((long)this.startMs), StringUtils.nullToEmptyNonDruidDataString((String)remoteAddress), new QueryStats(statsMap)));
        }
        catch (Exception ex) {
            log.error((Throwable)ex, "Unable to log query [%s]!", new Object[]{this.baseQuery});
        }
    }

    @Nullable
    public Query<?> getQuery() {
        return this.baseQuery;
    }

    public String getQueryId() {
        return this.baseQuery.getId();
    }

    public String threadName(String currThreadName) {
        return StringUtils.format((String)"%s[%s_%s_%s]", (Object[])new Object[]{currThreadName, this.baseQuery.getType(), this.baseQuery.getDataSource().getTableNames(), this.getQueryId()});
    }

    private boolean isSerializeDateTimeAsLong() {
        boolean shouldFinalize = QueryContexts.isFinalize(this.baseQuery, (boolean)true);
        return QueryContexts.isSerializeDateTimeAsLong(this.baseQuery, (boolean)false) || !shouldFinalize && QueryContexts.isSerializeDateTimeAsLongInner(this.baseQuery, (boolean)false);
    }

    public ObjectWriter newOutputWriter(QueryResource.ResourceIOReaderWriter ioReaderWriter) {
        return ioReaderWriter.getResponseWriter().newOutputWriter(this.getToolChest(), this.baseQuery, this.isSerializeDateTimeAsLong());
    }

    public QueryToolChest getToolChest() {
        if (this.state.compareTo(State.INITIALIZED) < 0) {
            throw new ISE("Not yet initialized", new Object[0]);
        }
        return this.toolChest;
    }

    private void transition(State from, State to) {
        if (this.state != from) {
            throw new ISE("Cannot transition from[%s] to[%s].", new Object[]{from, to});
        }
        this.state = to;
    }

    public static class QueryResponse {
        private final Sequence results;
        private final ResponseContext responseContext;

        private QueryResponse(Sequence results, ResponseContext responseContext) {
            this.results = results;
            this.responseContext = responseContext;
        }

        public Sequence getResults() {
            return this.results;
        }

        public ResponseContext getResponseContext() {
            return this.responseContext;
        }
    }

    static enum State {
        NEW,
        INITIALIZED,
        AUTHORIZING,
        AUTHORIZED,
        EXECUTING,
        UNAUTHORIZED,
        DONE;

    }
}

