package edu.ecompus.core.dao.impl;

import java.io.Serializable;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.hibernate.Query;
import org.hibernate.ScrollableResults;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.jdbc.Work;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import edu.ecompus.core.common.exception.ML2DaoException;
import edu.ecompus.core.dao.ML2GenericDao;
import edu.ecompus.core.entity.Page;



/**
 * DAO基层接口实现类
 * 
 * 
 * @param <T>
 * Project:               E-COMPUS CORE
 * Module ID:   <(模块)类编号，可以引用系统设计中的类编号>
 * Comments:  <对此类的描述，可以引用系统设计中的描述>
 * JDK version used:      <JDK1.7> 
 * @CopyRight CopyRright (c) 2015
 * @author JannyShao(邵建义) [ksgameboy@qq.com]
 * @since Date： 2015-3-28 上午9:38:55
 */
public class ML2TemplateGenericDao<Model, PK extends Serializable> implements ML2GenericDao<Model, PK> {
	
	protected final Logger logger = LoggerFactory.getLogger(getClass());
	/**
	 * JDBC模板
	 */
	@Autowired
	protected JdbcTemplate jdbcTemplate = null;
	
	/**
	 * Hibernate4 SessionFactory模板
	 */
	@Autowired
	protected SessionFactory sessionFactory;
	
	//TODO mybatis3模板
	
	protected final Class clazz;
	
	public ML2TemplateGenericDao(Class clazz)
	{
		this.clazz = clazz;
	}
	
	/**
     * gerCurrentSession 会自动关闭session，使用的是当前的session事务
     * 
     * @return
     */
    public Session getSession() {
    	if(sessionFactory==null){
    		throw new ML2DaoException("未配置SessionFactory,请查看XML文件! ");
    	}
        return sessionFactory.getCurrentSession();
    }
    
    /**
     * openSession 需要手动关闭session 意思是打开一个新的session
     * 
     * @return
     */
    public Session getNewSession() {
    	if(sessionFactory==null){
    		throw new ML2DaoException("未配置SessionFactory,请查看XML文件! ");
    	}
        return sessionFactory.openSession();
    }
	
    @SuppressWarnings("unchecked")
	@Override
	public Model get(PK id) {
		Session session = getSession();
        return (Model) session.get(clazz, id);
	}
	
	/**
	 * 创建clazz
	 * @param t
	 */
    @Override
	public void create(Model t)
	{
		try {
            Session session = getNewSession();
            session.save(t);
            session.flush();
            session.clear();
            session.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
	}

	/**
	 * 删除clazz
	 * @param t
	 */
    @Override
	public void delete(Model t)
	{
		Session session = getNewSession();
        session.delete(t);
        session.flush();
        session.clear();
        session.close();
	}
	
	/**
	 * 根据ID删除clazz
	 * @param id
	 */
    @Override
	public void deleteByPK(PK id){
		Session session = getNewSession();
        Object obj = session.get(clazz, id);
        session.delete(obj);
        flush();
        clear();
	}
    
	/**
	 * 根据IDs删除 clazz
	 * @param ids
	 */
    @Override
	public void deleteByList(PK[] ids){
		for (Serializable id : ids) {
            Object obj = getSession().get(clazz, id);
            if (obj != null) {
                getSession().delete(obj);
            }
        }
	}
	

	/**
	 * 更新clazz
	 * @param t
	 */
    @Override
	public void update(Model t)
	{
		Session session = getNewSession();
        session.update(t);
        session.flush();
        session.clear();
        session.close();
	}
	
	
	/**
	 * 得到全部数据
	 * @return
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public List getAllList() {
        String hql = "from " + clazz.getName();
        Session session = getSession();
        return session.createQuery(hql).list();
    }
	
	/**
     * 获取总数量
     * 
     * @param c
     * @return
     */
	@Override
	public Long getTotalCount() {
        Session session = getNewSession();
        String hql = "select count(id) from " + clazz.getName();
        Long count = (Long) session.createQuery(hql).uniqueResult();
        session.close();
        return count != null ? count.longValue() : 0;
    }


	/**
	 * 提交
	 */
	@Override
	public void flush() {
        getSession().flush();
    }

	/**
	 * 清除
	 */
	@Override
    public void clear() {
        getSession().clear();
    }
    
    /**
     * <执行Hql语句>
     * @param hqlString hql
     * @param values 不定参数数组
     */
    @Override
    public void queryHql(String hqlString, Object... values) {
        Query query = this.getSession().createQuery(hqlString);
        if (values != null)
        {
            for (int i = 0; i < values.length; i++)
            {
                query.setParameter(i, values[i]);
            }
        }
        query.executeUpdate();
    }


	@Override
	public void querySql(String sqlString, Object... values) {
		Query query = this.getSession().createSQLQuery(sqlString);
        if (values != null)
        {
            for (int i = 0; i < values.length; i++)
            {
                query.setParameter(i, values[i]);
            }
        }
        query.executeUpdate();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Model getByHQL(String hqlString, Object... values) {
		Query query = this.getSession().createQuery(hqlString);
        if (values != null)
        {
            for (int i = 0; i < values.length; i++)
            {
                query.setParameter(i, values[i]);
            }
        }
        return (Model) query.uniqueResult();
	}

	@SuppressWarnings("unchecked")
	@Override
	public Model getBySQL(String sqlString, Object... values) {
		Query query = this.getSession().createSQLQuery(sqlString);
        if (values != null)
        {
            for (int i = 0; i < values.length; i++)
            {
                query.setParameter(i, values[i]);
            }
        }
        return (Model) query.uniqueResult();
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<Model> getListByHQL(String hqlString, Object... values) {
		Query query = this.getSession().createQuery(hqlString);
        if (values != null)
        {
            for (int i = 0; i < values.length; i++)
            {
                query.setParameter(i, values[i]);
            }
        }
        return query.list();
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<Model> getListBySQL(String sqlString, Object... values) {
		Query query = this.getSession().createSQLQuery(sqlString);
        if (values != null)
        {
            for (int i = 0; i < values.length; i++)
            {
                query.setParameter(i, values[i]);
            }
        }
        return query.list();
	}

	@Override
	public List findListBySql(final String sql, final RowMapper map, final Object... values) {
		final List list = new ArrayList();
        // 执行JDBC的数据批量保存
        Work jdbcWork = new Work(){
            public void execute(Connection connection)
                throws SQLException
            {
                PreparedStatement ps = null;
                ResultSet rs = null;
                try
                {
                    ps = connection.prepareStatement(sql);
                    for (int i = 0; i < values.length; i++)
                    {
                        setParameter(ps, i, values[i]);
                    }
                     
                    rs = ps.executeQuery();
                    int index = 0;
                    while (rs.next())
                    {
                        Object obj = map.mapRow(rs, index++);
                        list.add(obj);
                    }
                }
                finally
                {
                    if (rs != null)
                    {
                        rs.close();
                    }
                    if (ps != null)
                    {
                        ps.close();
                    }
                }
            }
        };
        this.getSession().doWork(jdbcWork);
        return list;
	}

	@Override
	public Long countByHql(String hql, Object... values) {
		Query query = this.getSession().createQuery(hql);
        if(values != null){
            for(int i = 0; i < values.length; i++) {
                query.setParameter(i, values[i]);
            }
        }
        return (Long) query.uniqueResult();
	}

	@Override
	public Page<Model> findPageByFetchedHql(String hql, String countHql,
			int pageNo, int pageSize, Object... values) {
		Page<Model> retValue = new Page<Model>();
        Query query = this.getSession().createQuery(hql);
        if(values != null){
            for(int i = 0; i < values.length; i++) {
                query.setParameter(i, values[i]);
            }
        }
        int currentPage = pageNo > 1 ? pageNo : 1;
        
        //当前页码
        retValue.setPageNo(currentPage);
        //每页数量
        retValue.setPageSize(pageSize);
        //统计数量HSQL
        if (countHql == null)
        {
            ScrollableResults results = query.scroll();
            results.last();
            retValue.setTotalCount(results.getRowNumber() + 1);// 设置总记录数
        }
        else
        {
            Long count = countByHql(countHql, values);
            retValue.setTotalCount(count.intValue());
        }
        
        retValue.resetPageNo();
        List<Model> itemList = query.setFirstResult((currentPage - 1) * pageSize).setMaxResults(pageSize).list();
        if (itemList == null)
        {
            itemList = new ArrayList<Model>();
        }
        retValue.setResults(itemList);
         
        return retValue;
	}

	
	/**
     * 
     * 设置每行批处理参数
     * 
     * @param ps
     * @param pos ?占位符索引，从0开始
     * @param data
     * @throws SQLException
     * @see [类、类#方法、类#成员]
     */
    private void setParameter(PreparedStatement ps, int pos, Object data)
        throws SQLException
    {
        if (data == null)
        {
            ps.setNull(pos + 1, Types.VARCHAR);
            return;
        }
        Class dataCls = data.getClass();
        if (String.class.equals(dataCls))
        {
            ps.setString(pos + 1, (String)data);
        }
        else if (boolean.class.equals(dataCls))
        {
            ps.setBoolean(pos + 1, ((Boolean)data));
        }
        else if (int.class.equals(dataCls))
        {
            ps.setInt(pos + 1, (Integer)data);
        }
        else if (double.class.equals(dataCls))
        {
            ps.setDouble(pos + 1, (Double)data);
        }
        else if (Date.class.equals(dataCls))
        {
            Date val = (Date)data;
            ps.setTimestamp(pos + 1, new Timestamp(val.getTime()));
        }
        else if (BigDecimal.class.equals(dataCls))
        {
            ps.setBigDecimal(pos + 1, (BigDecimal)data);
        }
        else
        {
            // 未知类型
            ps.setObject(pos + 1, data);
        }
         
    }
}
