package edu.ecompus.core.dao;

import java.util.List;

import org.springframework.jdbc.core.RowMapper;

import edu.ecompus.core.entity.Page;


/**
 * DAO 基层接口类
 * <p>所有自定义Dao的顶级接口, 封装常用的增删查改操作
 * 可以通过Mybatis Generator Maven 插件自动生成Dao,
 * 也可以手动编码,然后继承ML2GenericDao 即可
 * <p>
 * @param <T> : 代表数据库中的表 映射的Java对象类型
 * @param <PK> : 代表对象的主键类型
 * Project:               E-COMPUS CORE
 * Module ID:   <(模块)类编号，可以引用系统设计中的类编号>
 * Comments:  <对此类的描述，可以引用系统设计中的描述>
 * JDK version used:      <JDK1.7> 
 * @CopyRight CopyRright (c) 2015
 * @author JannyShao(邵建义) [ksgameboy@qq.com]
 * @since Date： 2015-3-28 上午9:21:58
 */
public abstract interface ML2GenericDao<Model, PK> {
	
	/**
	 * 查询对象
	 * @param id	查询编号
	 * @return
	 */
	public abstract Model get(PK id);
	
	/**
	 * 创建对象
	 * @param obj
	 */
	public abstract void create(Model model);

	/**
	 * 删除对象
	 * @param obj
	 */
	public abstract void delete(Model model);

	/**
	 * 通过主键删除对象
	 * @param obj
	 */
	public abstract void deleteByPK(PK id);
	
	/**
	 * 根据多个主键删除对象
	 * @param ids
	 */
	public abstract void deleteByList(PK[] ids);
	
	/**
	 * 更新对象
	 * @param obj
	 */
	public abstract void update(Model model);
	
	
	/**
	 * 读取所有列表
	 * @return
	 */
	public abstract List getAllList();
	
	/**
	 * 统计所有数量
	 * @return
	 */
	public abstract Long getTotalCount();
	
	/**
     * <执行Hql语句>
     * @param hqlString hql
     * @param values 不定参数数组
     */
    public abstract void queryHql(String hqlString, Object... values); 
     
    /**
     * <执行Sql语句>
     * @param sqlString sql
     * @param values 不定参数数组
     */
    public abstract void querySql(String sqlString, Object... values); 
 
    /**
     * <根据HQL语句查找唯一实体>
     * @param hqlString HQL语句
     * @param values 不定参数的Object数组
     * @return 查询实体
     */
    public abstract Model getByHQL(String hqlString, Object... values);
 
    /**
     * <根据SQL语句查找唯一实体>
     * @param sqlString SQL语句
     * @param values 不定参数的Object数组
     * @return 查询实体
     */
    public abstract Model getBySQL(String sqlString, Object... values);
 
    /**
     * <根据HQL语句，得到对应的list>
     * @param hqlString HQL语句
     * @param values 不定参数的Object数组
     * @return 查询多个实体的List集合
     */
    public abstract List<Model> getListByHQL(String hqlString, Object... values);
 
    /**
     * <根据SQL语句，得到对应的list>
     * @param sqlString HQL语句
     * @param values 不定参数的Object数组
     * @return 查询多个实体的List集合
     */
    public abstract List<Model> getListBySQL(String sqlString, Object... values);
     
    /**
     * 由sql语句得到List
     * @param sql
     * @param map
     * @param values
     * @return List
     */
    public List findListBySql(final String sql, final RowMapper map, final Object... values);
 
 
    /**
     * <根据HQL得到记录数>
     * @param hql HQL语句
     * @param values 不定参数的Object数组
     * @return 记录总数
     */
    public abstract Long countByHql(String hql, Object... values);
     
    /**
     * <HQL分页查询>
     * @param hql HQL语句
     * @param countHql 查询记录条数的HQL语句
     * @param pageNo 下一页
     * @param pageSize 一页总条数
     * @param values 不定Object数组参数
     * @return PageResults的封装类，里面包含了页码的信息以及查询的数据List集合
     */
    public abstract Page<Model> findPageByFetchedHql(String hql, String countHql, int pageNo, int pageSize, Object... values);
    
    /**
     * 手动提交
     */
    public abstract void flush();
    
    /**
     * 清除表?
     */
    public abstract void clear();
}
