package edu.ecompus.core.filter;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.core.io.Resource;

import edu.ecompus.core.util.FileUtil;
import edu.ecompus.core.util.GZipUtil;


/**
 * 文件缓存过滤器
 * @author Janny E-mail:ksgameboy@qq.com
 * @version 1.0
 * @since 1.0
 */
public class FileCacheFilter extends AbstractCacheFilter {
	private String root;
	private final String SUFFIX = ".html";
	
	public final void setFileDir(Resource dir){
		try
		{
			File f = dir.getFile();
			f.mkdirs();
			if (!f.isDirectory())
				throw new IllegalArgumentException((new StringBuilder()).append("Invalid directoy: ").append(f.getPath()).toString());
			if (!f.canWrite())
				throw new IllegalArgumentException((new StringBuilder()).append("Cannot write to directory: ").append(f.getPath()).toString());
			root = f.getPath();
			if (!root.endsWith("/") && !root.endsWith("//"))
				root = (new StringBuilder()).append(root).append("/").toString();
		}
		catch (IOException e)
		{
			throw new IllegalArgumentException(e);
		}
	}
	
	public void afterPropertiesSet() throws Exception
	{
		super.afterPropertiesSet();
		if (!(new File(root)).isDirectory())
			throw new IllegalArgumentException((new StringBuilder()).append("No directory: ").append(root).toString());
		else
			return;
	}
	
	@SuppressWarnings("unchecked")
	public void remove(String url, Map parameters)
	{
		String key = getKey(HttpServletRequestFactory.create(url, parameters));
		if (key != null)
			FileUtil.removeFile(key2File(key));
	}
	
	/**
	 * 过滤
	 */
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
		throws IOException, ServletException
	{
		HttpServletRequest httpRequest = (HttpServletRequest)request;
		String key = getKey(httpRequest);
		if (key == null)
		{
			chain.doFilter(request, response);
		} else
		{
			File file = key2File(key);
			if (file.isFile())
			{
				logger.info((new StringBuilder()).append("Load cached file from: ").append(file.getPath()).toString());
				HttpServletResponse httpResponse = (HttpServletResponse)response;
				httpResponse.setContentType(getContentType());
				httpResponse.setHeader("Content-Encoding", "gzip");
				httpResponse.setContentLength((int)file.length());
				FileUtil.readFile(file, httpResponse.getOutputStream());
			} else
			{
				logger.info("Cannot find cached file. Create new cached file...");
				HttpServletResponse httpResponse = (HttpServletResponse)response;
				CachedResponseWrapper wrapper = new CachedResponseWrapper(httpResponse);
				chain.doFilter(request, wrapper);
				if (wrapper.getStatus() == 200)
				{
					byte data[] = GZipUtil.gzip(wrapper.getResponseData());
					FileUtil.writeFile(file, data);
					logger.info("Page is cached to file.");
					httpResponse.setContentType(getContentType());
					httpResponse.setHeader("Content-Encoding", "gzip");
					httpResponse.setContentLength(data.length);
					httpResponse.getOutputStream().write(data);
				} else
				{
					logger.warn("Cannot cache page because response code is not 200.");
				}
			}
		}
	}
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	
	private File key2File(String key)
	{
		int hash = key.hashCode();
		int dir1 = (hash & 0xff00) >> 8;
		int dir2 = hash & 0xff;
		String dir = (new StringBuilder()).append(root).append(dir1).append("/").append(dir2).toString();
		File fdir = new File(dir);
		if (!fdir.isDirectory() && !fdir.mkdirs())
			return null;
		else
			return new File((new StringBuilder()).append(dir).append("/").append(encode(key)).append(SUFFIX).toString());
	}
	
	private String encode(String key)
	{
		try{
			return URLEncoder.encode(key, "UTF-8");
		}catch(UnsupportedEncodingException e){
			throw new RuntimeException(e);
		}
		
	}
	
}
