package edu.ecompus.cas.mail;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.mail.Address;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.exception.VelocityException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.mail.MailException;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.mail.javamail.MimeMessagePreparator;
import org.springframework.ui.velocity.VelocityEngineUtils;

import edu.ecompus.cas.util.StringUtil;

/**
 * 类描述：邮件发送模板
 * 
 * @author wsh
 * @date: 日期：2013-11-22 时间：上午10:58:21
 * @version 1.0
 */
public class VelocityMailService extends JavaMailSenderImpl implements JavaMailSender {

	private Properties mailHeaders = new Properties();

	/** 邮件发送者，包括发送者姓名和地址，用于设置在邮件的from栏目中 */
	private String from;
	
	/** 邮件主题 */
	private String subject;
	
	/** 邮件内容模板地址/名称 */
	private String templateName;
	
	//发送用户列表
	private ArrayList<String> userNameList;
	private ArrayList<String> passwordList;
	private int currentMailId = 0;
	//多个接收邮箱
	private ArrayList<String> mutiUser;
	//附件列表
	private Map<String, Object> attachments;
	//行内附件
	private Map<String, Object> inlines;
	
	
	/** velocity引擎 */
	@Autowired
	private VelocityEngine velocityEngine;
	/**
	 * 客服邮箱发送器@see applicationContext-email.xml
	 */
	@Autowired
	private JavaMailSender mailSender;
	
	/**
	 * Me邮箱发送器@see applicationContext-email.xml
	 */
	@Autowired
	private JavaMailSender meMailSender;
	
	/**
	 * 邮箱群发发送器@see applicationContext-email.xml
	 */
	@Autowired
	private JavaMailSender muleMailSender;

	/**
	 * 使用提供的数据，套入velocity模板，生成最后文本信息。
	 * 
	 * 大部分情况下，这个信息应该就是邮件的内容。
	 */
	public String renderText(Map model) throws VelocityException {
		
		// 防止乱码，需设置为UTF-8
		return VelocityEngineUtils.mergeTemplateIntoString(getVelocityEngine(), getTemplateName(), "utf-8", model);
	}

	public Properties getMailHeaders() {
		return mailHeaders;
	}

	/**
	 * 
	 * 发送Mime邮件简易方法。
	 * 
	 * 以Mime的方式发送邮件，这主要是为能够支持发送html或类似功能（非简单文本）的邮件
	 * 
	 * @param nameOfTo
	 *            邮件接收收人姓名 或 昵称
	 * @param emailOfTo
	 *            邮件接收人邮件地址
	 * @param model
	 *            邮件velocity模板中变量的值
	 * @throws MailException
	 */
	public boolean sendMime(String nameOfTo, String emailOfTo,
			Map<String, Object> model){
		try{
			return sendMime(nameOfTo + "<" + emailOfTo + ">", model);
		}catch(MailException e){
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * 发送Mime邮件简易方法。
	 * 
	 * 以Mime的方式发送邮件，这主要是为能够支持发送html或类似功能（非简单文本）的邮件
	 * 
	 * @param to
	 *            邮件接收人邮件地址以及可能的收件人姓名或昵称
	 * @param model
	 *            邮件velocity模板中变量的值
	 */
	public boolean sendMime(String to, Map<String, Object> model) {
		    try{
				return sendMime(mergeSimpleMessage(to, model));
		    }catch(MailException e){
		    	e.printStackTrace();
		    	return false;
		    }
	}

	/**
	 * 发送Mime邮件简易方法。
	 * 
	 * 以Mime的方式发送SimpleMailMessage，这主要是为能够支持发送html或类似功能（非简单文本）的邮件
	 * 
	 * @param simpleMessage
	 * @throws MailException
	 */
	public boolean sendMime(SimpleMailMessage simpleMessage){
		try{
		    mailSender.send(toMimeMessage(simpleMessage));
		    return true;
		}catch(MailException e){
			e.printStackTrace();
			return false;
		}
	}
	
	/**
	 * 发送邮件(多接收人、带附件）
	 * @param to
	 * @param model
	 * @return
	 */
	public boolean sendMail(Map<String, Object> model){
		try{
			/* 3.发送邮件(JavaMailSender) */
			MimeMessage message = mailSender.createMimeMessage();
			MimeMessageHelper helper = new MimeMessageHelper(message, true, "UTF-8");// 处理中文编码
			
			//通过模板VM文件生成邮件HTML内容
			String text = null;
			try {
				text = renderText(model);
			} catch (VelocityException e) {
				e.printStackTrace();
			}
			// 信息
			helper.setSubject(getSubject());									// 主题
			helper.setFrom(getFrom()); 											// 发件人
	        helper.setText(text, true);											// 内容(HTML)
	        
	        //行内附件列表
	        if(inlines!=null && inlines.size()>0){
	        	for (String inline : inlines.keySet()) {
	        	    File inlineFile = (File) inlines.get(inline);
	        	    helper.addInline(inline, inlineFile);
	        	}
	        }
	        //附件列表
	        if(inlines!=null && attachments.size()>0){
	        	for (String attachment : attachments.keySet()) {
	        	    File attachmentFile = (File) attachments.get(attachment);
	        	    helper.addAttachment(attachment, attachmentFile);
	        	}
	        }
	        
	        //多个账号
	        if(mutiUser!=null && mutiUser.size()>0){
	        	InternetAddress[] to_mail = new InternetAddress[mutiUser.size()];
	        	for (int i = 0; i < mutiUser.size(); i++) { // 设置接收邮件人的地址
	            	to_mail[i] = new InternetAddress(mutiUser.get(i));
	            }
	        	 helper.setTo(to_mail);
	        }
	        
//	        String contentId = "inline";
//	        Resource resource = new ClassPathResource("inline.jpg");
//	        String attachmentFilename = "attachment";
//	        InputStreamSource inputStreamSource = new ClassPathResource("attachment.txt");
//	        helper.addInline(contentId, resource);											// 附件(行内)
//			helper.addAttachment(attachmentFilename, inputStreamSource);					// 附件
			
			// 发送
			mailSender.send(message);
		    return true;
		}catch(Exception e){
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * 设置非空mime hader，一般可以利用此设置contentType等
	 * 
	 * @param mailHeaders
	 */
	public void setMailHeaders(Properties mailHeaders) {
		this.mailHeaders = mailHeaders;
	}

	public String getFrom() {
		return this.from;
	}

	public Address getFromAddress() throws AddressException {
		return new InternetAddress(this.from);
	}

	/**
	 * 设置邮件发送者，包括发送者姓名和地址，用于设置在邮件的from栏目中
	 * 
	 * @param from
	 */
	public void setFrom(String from) {
		this.from = from;
	}

	public String getSubject() {
		return subject;
	}

	public void setSubject(String subject) {
		this.subject = subject;
	}

	public String getTemplateName() {
		return templateName;
	}

	/**
	 * 设置邮件内容模板地址/名称
	 * 
	 * @param templateName
	 */
	public void setTemplateName(String templateName) {
		this.templateName = templateName;
	}

	public VelocityEngine getVelocityEngine() {
		return velocityEngine;
	}

	public void setVelocityEngine(VelocityEngine velocityEngine) {
		this.velocityEngine = velocityEngine;
	}

	/**
	 * 获取下层配置的mail发送器，发送邮件, 该mail发送器必须是JavaMailSender，否则抛出CaseException。
	 * 
	 * @return
	 */
	public JavaMailSender getJavaMailSender() {
		return (JavaMailSender) mailSender;
	}

	/**
	 * 设置mail发送器
	 * 
	 * @param mailEngine
	 */
	public void setMailSender(JavaMailSender mailSender) {
		this.mailSender = mailSender;
	}
	
	@Override
	public void setUsername(String username) {
		if (userNameList == null)
			userNameList = new ArrayList<String>();
		// split usernmae in ,
		String[] userNames = username.split(",");
		if (userNames != null) {
			for (String user : userNames) {
				userNameList.add(user);
			}
		}
		// before send super.setXxx()		// accounts
		super.setUsername(userNameList.get(currentMailId));
	}
	
	@Override
	public void setPassword(String password) {
		if (passwordList == null)
			passwordList = new ArrayList<String>();
		// split password in ,
		String[] passwords = password.split(",");
		if (passwords != null) {
			for (String pw : passwords) {
				passwordList.add(pw);
			}
		}
		// before send super.setXxx() 	// accounts
		super.setPassword(passwordList.get(currentMailId));
	}
	
	@Override
	protected void doSend(MimeMessage[] mimeMessage, Object[] object) throws MailException {
		super.doSend(mimeMessage, object);
		
		// Next Accounts
		currentMailId = (currentMailId + 1) % userNameList.size();
		super.setUsername(userNameList.get(currentMailId));
		super.setPassword(passwordList.get(currentMailId));
	}
	
	
	
	
	/**
	 * 产生简单信息
	 * 
	 * @param to
	 * @param model
	 * @return
	 */
	public SimpleMailMessage mergeSimpleMessage(String to,
			Map<String, Object> model) {
		// render text of mail from velocity template with the data
		String text = null;
		try {
			text = renderText(model);
		} catch (VelocityException e) {
			e.printStackTrace();
		}

//		System.out.println(text);

		// mail message setting
		SimpleMailMessage message = new SimpleMailMessage();
		message.setSubject(getSubject());
		message.setFrom(getFrom());
		message.setTo(to);
		message.setText(text);
		return message;
	}

	/**
	 * 产生html文本信息
	 * 
	 * @param simpleMailMessage
	 * @return
	 */
	public MimeMessagePreparator toMimeMessage(final SimpleMailMessage simpleMailMessage) {
		MimeMessagePreparator preparator = new MimeMessagePreparator() {
			public void prepare(MimeMessage mimeMessage) throws Exception {
				// TODO Auto-generated method stub
				MimeMessageHelper message = new MimeMessageHelper(mimeMessage,		true, "utf-8");
				message.setTo(simpleMailMessage.getTo());
				message.setSubject(simpleMailMessage.getSubject());
				message.setText(simpleMailMessage.getText(), true);
				message.setFrom(simpleMailMessage.getFrom());
			}
		};

		return preparator;
	}
	
	/**
	 * 添加附件
	 * @param attachmentFilename
	 * @param file
	 */
	public void addAttachments(String attachmentFilename, File file) {
		if (attachments == null)
			attachments = new HashMap<String, Object>();
		
		
		if(attachmentFilename==null || "".equals(attachmentFilename)){
			return;
		}
		
		if(file==null){
			return;
		}
		
		attachments.put(attachmentFilename, file);
	}
	
	/**
	 * 添加附件列表
	 * @param files
	 */
	public void addAttachmentsList(Map<String, Object> files) {
		if (attachments == null)
			attachments = new HashMap<String, Object>();
		
		attachments.putAll(files);
	}

	/**
	 * 添加行内附件
	 * @param attachmentFilename
	 * @param file
	 */
	public void addInlines(String attachmentFilename, File file) {
		if (inlines == null)
			inlines = new HashMap<String, Object>();
		
		
		if(attachmentFilename==null || "".equals(attachmentFilename)){
			return;
		}
		
		if(file==null){
			return;
		}
		
		inlines.put(attachmentFilename, file);
	}
	
	/**
	 * 添加行内附件列表
	 * @param attachmentFilename
	 * @param file
	 */
	public void addInlinesList(Map<String, Object> files) {
		if (inlines == null)
			inlines = new HashMap<String, Object>();
		
		inlines.putAll(files);
	}
	
	/**
	 * 添加接收邮箱
	 * @param userMail
	 */
	public void addToUser(String userMail){
		if (mutiUser == null)
			mutiUser = new ArrayList<String>();
		
		mutiUser.add(userMail);
	}
	/**
	 * 添加接收邮箱列表
	 * @param mutiUser
	 */
	public void addUserList(List<String> mutiUser){
		if (this.mutiUser == null)
			this.mutiUser = new ArrayList<String>();
		
		this.mutiUser.clear();
		this.mutiUser.addAll(mutiUser);
	}
	
	
	/**
	 * 根据Me邮箱发送邮件(多接收人、带附件）
	 * @param to
	 * @param model
	 * @return
	 */
	public boolean sendMeMail(Map<String, Object> model){
		try{
			/* 3.发送邮件(JavaMailSender) */
			MimeMessage message = meMailSender.createMimeMessage();
			MimeMessageHelper helper = new MimeMessageHelper(message, true, "UTF-8");// 处理中文编码
			
			//通过模板VM文件生成邮件HTML内容
			String text = null;
			try {
				text = renderText(model);
			} catch (VelocityException e) {
				e.printStackTrace();
			}
			// 信息
			helper.setSubject(getSubject());									// 主题
			helper.setFrom(getFrom()); 											// 发件人
	        helper.setText(text, true);											// 内容(HTML)
	        
	        //行内附件列表
	        if(inlines!=null && inlines.size()>0){
	        	for (String inline : inlines.keySet()) {
	        	    File inlineFile = (File) inlines.get(inline);
	        	    helper.addInline(inline, inlineFile);
	        	}
	        }
	        //附件列表
	        if(inlines!=null && attachments.size()>0){
	        	for (String attachment : attachments.keySet()) {
	        	    File attachmentFile = (File) attachments.get(attachment);
	        	    helper.addAttachment(attachment, attachmentFile);
	        	}
	        }
	        
	        //多个账号
	        if(mutiUser!=null && mutiUser.size()>0){
	        	InternetAddress[] to_mail = new InternetAddress[mutiUser.size()];
	        	for (int i = 0; i < mutiUser.size(); i++) { // 设置接收邮件人的地址
	            	to_mail[i] = new InternetAddress(mutiUser.get(i));
	            }
	        	 helper.setTo(to_mail);
	        }
	        
			
			// 发送
	        meMailSender.send(message);
		    return true;
		}catch(Exception e){
			e.printStackTrace();
			return false;
		}
	}
	
	/**
	 * 根据shimi@36uk.cn邮箱发送邮件(多接收人、带附件）
	 * @param to
	 * @param model
	 * @return
	 */
	public boolean sendMuleMail(String toEmail , Map<String, Object> model){
		try{
			/* 3.发送邮件(JavaMailSender) */
			MimeMessage message = meMailSender.createMimeMessage();
			MimeMessageHelper helper = new MimeMessageHelper(message, true, "UTF-8");// 处理中文编码
			
			//通过模板VM文件生成邮件HTML内容
			String text = null;
			try {
				text = renderText(model);
			} catch (VelocityException e) {
				e.printStackTrace();
			}
			// 信息
			helper.setSubject(getSubject());									// 主题
			helper.setFrom(getFrom()); 											// 发件人
	        helper.setText(text, true);											// 内容(HTML)
	        
	        //行内附件列表
	        if(inlines!=null && inlines.size()>0){
	        	for (String inline : inlines.keySet()) {
	        	    File inlineFile = (File) inlines.get(inline);
	        	    helper.addInline(inline, inlineFile);
	        	}
	        }
	        //附件列表
	        if(inlines!=null && attachments.size()>0){
	        	for (String attachment : attachments.keySet()) {
	        	    File attachmentFile = (File) attachments.get(attachment);
	        	    helper.addAttachment(attachment, attachmentFile);
	        	}
	        }
	        
	        //多个账号
	        if(mutiUser!=null && mutiUser.size()>0){
	        	InternetAddress[] to_mail = new InternetAddress[mutiUser.size()];
	        	for (int i = 0; i < mutiUser.size(); i++) { // 设置接收邮件人的地址
	            	to_mail[i] = new InternetAddress(mutiUser.get(i));
	            }
	        	 helper.setTo(to_mail);
	        }else if(!StringUtil.isEmpty(toEmail)){
	        	helper.setTo(toEmail);
	        }
	        
			
			// 发送
	        muleMailSender.send(message);
		    return true;
		}catch(Exception e){
			e.printStackTrace();
			return false;
		}
	}
	
}