package com.taobao.config.client;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Map;
import java.util.UUID;

import com.taobao.config.client.bean.InstanceMetaData;
import com.taobao.config.client.exception.ExcEvent;
import com.taobao.config.client.exception.ExcEventDispatch;
import com.taobao.config.client.exception.ExcListener;
import com.taobao.config.client.exception.ExcRole;
import com.taobao.config.client.exception.ExcType;
import com.taobao.config.client.utils.AttributeUtils;
import com.taobao.config.client.utils.StringUtils;
import com.taobao.config.common.ErrorType;
import com.taobao.config.common.Revision;
import com.taobao.config.common.protocol.AttributeElement;
import com.taobao.config.common.protocol.ProtocolPackage;
import com.taobao.config.common.protocol.PublisherDataElement;
import com.taobao.config.common.protocol.PublisherRegReqPacket;
import com.taobao.config.common.protocol.UserDataResultElement;
import com.taobao.middleware.logger.Logger;

/**
 * @author wufeng
 *
 */
public class DefaultPublisher<T extends Serializable> extends DefaultDataClient implements Publisher<T> {

    private final PublisherRegReqPacket regRequest;
    private volatile Object data = null;
    private volatile Revision revision = initialRevision;
    private Revision ackedRevision = initialRevision;

    private final String env;
    private static final Logger log = ConfigClientLogger.getLogger(DefaultPublisher.class);
    private static final Revision initialRevision = new Revision(1);

    @Override
    public void publish(T datum) {
        if (null == datum) {
            throw new IllegalArgumentException("cannot publish null");
        }
        if (isDisable())
            throw new IllegalStateException("Unregistered publisher cannot be reused.");
        try {
            new ObjectOutputStream(new OutputStream() {

                @Override
                public void write(int b) throws IOException {}
            }).writeObject(datum);
        } catch (IOException e) {
            throw new RuntimeException("Not serializable datum", e);
        }
        //ݳȿ
        if (datum.getClass() == String.class) {
            String tmp = (String) datum;
            if (tmp != null && tmp.length() > ConfigClientPerfCtrl.dataContentLengthMax) {
                throw new IllegalArgumentException("publisher content length bigger than " + ConfigClientPerfCtrl.dataContentLengthMax + " :" + tmp);
            }
        }
        doPublish(datum);
    }

    private synchronized void doPublish(Object datum) {
        data = datum;
        revision = revision.getUpdatedRevision();
        String dataId = getDataId();
        String publisherId = getClientId();
        if (datum == null) {
            log.info("[Publish] " + dataId + " (rev." + revision + ", by " + publisherId + "): (NULL)");
        } else {
            log.info("[Publish] dataId=" + dataId + ", clientId=" + publisherId + ", datumId=" + registration.getDatumId() + ", tenant=" + registration.getTenant() + ", rev=" + revision.getRevision()+ ", env=" + env);
        }
        signalWorker();
    }

    @SuppressWarnings("unchecked")
    public PublisherRegistration<T> getRegistration() {
        return (PublisherRegistration<T>) registration;
    }

    DefaultPublisher(PublisherRegistration<T> registration) {
        super(registration);
        //set env value
        Object center = registration.getLocalAttribute(LocalAttribute.ATTRIBUTE_CENTER);
        Object server = registration.getLocalAttribute(LocalAttribute.ATTRIBUTE_SERVER);
        if (center != null) {
            env = (String) center;
        } else if (server != null) {
            env = (String) server;
        } else {
            env = "default";
        }
        // For asynchronous register
        String datumId = registration.getDatumId();
        this.regRequest = new PublisherRegReqPacket(registration.getDataId(), registration.getClientId(), datumId != null ? datumId : UUID.randomUUID().toString()); // UUID as default datum ID.
        for (Map.Entry<String, Serializable> attribute : registration.getAllAttributes()) {
            regRequest.setAttribute(attribute.getKey(), attribute.getValue());
        }
        //pubCount
        super.serverMgr.incrementPubCount();
    }

    public synchronized void unregister() {
        if (super.setDisable()) {
            revision = revision.getUpdatedRevision(); // 汾Ų
            //pubCount
            super.serverMgr.decrementPubCount();
            signalWorker();
        }
    }

    // Implement interface InternalPublisher
    public synchronized Revision revision() {
        return revision;
    }

    public synchronized void setRevision(Revision revision) {
        this.revision = revision;
    }

    // Implement interface Deliverable
    public synchronized boolean isDone() {
        // Registered and published data is up to date.
        return isRegistered() && ackedRevision.equals(revision);
    }

    @Override
    ServerListManager initServerManager() {
        String server = registration.getLocalAttribute(LocalAttribute.ATTRIBUTE_SERVER);
        String center = registration.getLocalAttribute(LocalAttribute.ATTRIBUTE_CENTER);
        InstanceMetaData instanceMetaData = registration.getInstanceMetaData();

        // ȼCenter>Server עͬʱ!Center!Server ,!ServerЧ
        if (StringUtils.isNotBlank(center)) {
            return ServerListManagerFactory.getServerListManager(LocalAttribute.ATTRIBUTE_CENTER, center,
                    LocalConfigInfo.SERVER_CON_PUB_TYPE, instanceMetaData);
        } else if(StringUtils.isNotBlank(server)) {
            return ServerListManagerFactory.getServerListManager(LocalAttribute.ATTRIBUTE_SERVER, server,
                    LocalConfigInfo.SERVER_CON_PUB_TYPE, instanceMetaData);
        } else {
            return ServerListManagerFactory.getDefaultServerListManager(LocalConfigInfo.SERVER_CON_PUB_TYPE, instanceMetaData);
        }
    }

    @Override
    synchronized boolean isSynchronized() {
        /**
         * 1.״̬ && δע && ݰ汾Ϊʼͬûpub
         * 2.״̬ && ע && ûб䶯
         */
        return (isEnable() && !isRegistered() && revision.equals(initialRevision)) // δݵ״̬
                || (isEnable() && isRegistered() && revision.equals(ackedRevision));
    }

    @Override
    public synchronized void addPackage(ProtocolPackage packagee) {
        if (cleanUpIfNecessary()) {
            return;
        }

        if (!isRegistered()) {
            packagee.addElement(regRequest);
            log.info("[Deliver] " + registration);
        }
        /***/
        packagee.addElement(new AttributeElement(AttributeElement.LANG, "java"));
        /**Ӱ汾*/
        packagee.addElement(new AttributeElement(AttributeElement.VERSION, LocalConfigInfo.getVersion()));
        InstanceMetaData instanceMetaData = registration.getInstanceMetaData();
        AttributeUtils.setInstanceMetaInPackage(packagee, instanceMetaData);

        String dataId = registration.getDataId();
        String publisherId = registration.getClientId();
        PublisherDataElement element = new PublisherDataElement(dataId, publisherId, revision);
        if (isDisable()) {
            // Empty list act as unregister request.
        } else {
            element.addData(Arrays.asList(data));
        }
        element.needAck = true;
        //TODO xxc
        //add spas and tenant attributeElement,server must process this element before PublisherDataElement
        packagee.addElement(new AttributeElement(AttributeElement.ATTRIBUTE_TENANT, "true"));
        packagee.addElement(new AttributeElement(AttributeElement.ATTRIBUTE_METADATA_GROUP, "true"));

        packagee.addElement(element);
        log.info("[Deliver] data. dataId=" + dataId + ", clientId=" + publisherId + ", datumId=" + registration.getDatumId() + ",tenant=" + this.getRegistration().getTenant() + ", rev=" + revision + ", env=" + env
                + (isDisable() ? ", unregister" : ""));
    }

    // 汾: revision >= recvRev >= ackRev
    public synchronized void handleDataResult(UserDataResultElement element) {
        String message = null;
        if (isNotRegistered()) {
            message = "[Publish-error] Received result of data not yet registered!";
            log.error("%s", message);
            return;
        }
        // Validate data ID.
        String myDataId = getDataId();
        if (!element.dataId.equals(myDataId)) {
            message = "[Publish-error] Data ID of received user data result does not match with the publisher! (" + element.dataId + " != " + myDataId + ")";
            log.error("%s", message);
            return;
        }
        // Check result
        Revision receivedRevision = element.revision;
        if (null != element.error) {
            if (ErrorType.SPAS_VERIFY_FAILURE.getIdentifyCode().equals(element.error.getMessage())) {
                //֤ʧܺ£ƳߣֹԷע֤ʧҲﴦreceivedRevision 3 Ϊ ע?
                PublisherRegistrar.remove(this);
                String errorMessage =
                        "[Publish or unregister error] spas-authentication-failed! dataId:" + myDataId + " group:" + this.getRegistration().getGroup() + " erorr:" + element.error
                                + " receivedRevision:" + receivedRevision + " tenant:" + this.getRegistration().getTenant();
                log.error("%s", errorMessage);
                ExcEventDispatch.fireEvent(new ExcEvent(this.getRegistration().getDataId(),//
                        this.getRegistration().getGroup(),//
                        this.getRegistration().getTenant(), //  
                        ExcType.PUB_PERMISSION, //
                        errorMessage));
            } else {
                message = "[Publish-error] Data " + myDataId + "(" + receivedRevision + ") published by " + getClientId() + " is refused with error: " + element.error;
                log.error("%s", message);
                onServerDisconnected();
                ExcEventDispatch.fireEvent(new ExcEvent(this.getRegistration().getDataId(),//
                        this.getRegistration().getGroup(),//
                        this.getRegistration().getTenant(), //  
                        ExcType.UKNOWN, //
                        message));
            }
            return;
        }
        // Validate revision
        if (receivedRevision.isNewerThan(revision)) {
            log.warn("[Publish-error] " + myDataId + ", recvRev." + receivedRevision + " > rev." + revision);
            return;
        }
        if (receivedRevision.isOlderThan(ackedRevision)) {
            log.warn("[Publish-error] " + myDataId + ", recvRev." + receivedRevision + " < ackRev." + ackedRevision);
            return;
        }
        if (isDisable() && receivedRevision.equals(revision)) {
            PublisherRegistrar.remove(this);
            log.info("[unregister-ok] dataId=" + myDataId + ", clientId=" + getClientId() + ",tenant=" + this.getRegistration().getTenant() + ", env=" + env);
        } else {
            log.info("[Publish-ok] dataId=" + myDataId + ", clientId=" + getClientId() + ", datumId=" + registration.getDatumId() + ",tenant=" + this.getRegistration().getTenant() + ", rev="
                    + receivedRevision + ", env=" + env);
        }
        ackedRevision = receivedRevision;
    }

    public String getEnv() {
        return env;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Publisher (" + getClientId() + " for " + getDataId() + ")");
        sb.append("Tenant:" + registration.getTenant());
        String group = registration.getGroup();
        if (group != null) {
            sb.append(" in group " + group);
        }
        sb.append(" in env " + env);
        return sb.toString();
    }

    @Override
    public void addExcListener(ExcListener listener) {
        ExcEventDispatch.addListener(this.getRegistration().getDataId(),//
                this.getRegistration().getGroup(), //
                this.getRegistration().getTenant(),//
                ExcRole.PUB, listener);
    }
}
