/*
 * Copyright 2008-2009 SpringSource
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.springsource.bundlor.support.contributors;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.springframework.util.StringUtils;

import com.springsource.bundlor.support.ManifestReader;
import com.springsource.bundlor.support.PartialManifestModifier;
import com.springsource.bundlor.support.TemplateHeaderReader;
import com.springsource.bundlor.support.partialmanifest.ReadablePartialManifest;
import com.springsource.bundlor.util.MatchUtils;
import com.springsource.bundlor.util.SimpleParserLogger;
import com.springsource.util.osgi.manifest.parse.HeaderDeclaration;
import com.springsource.util.osgi.manifest.parse.HeaderParser;
import com.springsource.util.osgi.manifest.parse.HeaderParserFactory;
import com.springsource.util.parser.manifest.ManifestContents;

/**
 * An implementation of {@link PartialManifestModifier} that removes excluded imports and exports from the partial
 * manifest
 * <p />
 * 
 * <strong>Concurrent Semantics</strong><br />
 * 
 * Threadsafe
 * 
 * @author Ben Hale
 */
public final class ExcludedImportAndExportPartialManifestModifier implements ManifestReader, PartialManifestModifier, TemplateHeaderReader {

    private static final String ATTR_EXCLUDED_EXPORTS = "Excluded-Exports";

    private static final String ATTR_EXCLUDED_IMPORTS = "Excluded-Imports";

    private final List<String> excludedImports = new ArrayList<String>();

    private final Object excludedImportsMonitor = new Object();

    private final List<String> excludedExports = new ArrayList<String>();

    private final Object excludedExportsMonitor = new Object();

    public void readJarManifest(ManifestContents manifest) {
        // Nothing to do
    }

    public void readManifestTemplate(ManifestContents manifestTemplate) {
        synchronized (excludedImportsMonitor) {
            String value = manifestTemplate.getMainAttributes().get(ATTR_EXCLUDED_IMPORTS);
            List<HeaderDeclaration> headers = parseTemplate(value);
            for (HeaderDeclaration header : headers) {
                excludedImports.add(header.getNames().get(0));
            }
        }
        synchronized (excludedExportsMonitor) {
            String value = manifestTemplate.getMainAttributes().get(ATTR_EXCLUDED_EXPORTS);
            List<HeaderDeclaration> headers = parseTemplate(value);
            for (HeaderDeclaration header : headers) {
                excludedExports.add(header.getNames().get(0));
            }
        }
    }

    private List<HeaderDeclaration> parseTemplate(String template) {
        if (StringUtils.hasText(template)) {
            HeaderParser parser = HeaderParserFactory.newHeaderParser(new SimpleParserLogger());
            return parser.parseHeader(template);
        }
        return new ArrayList<HeaderDeclaration>(0);
    }

    public void modify(ReadablePartialManifest partialManifest) {
        synchronized (excludedImportsMonitor) {
            removeExcludedPackages(partialManifest.getImportedPackages(), excludedImports);
        }
        synchronized (excludedExportsMonitor) {
            removeExcludedPackages(partialManifest.getExportedPackages(), excludedExports);
        }
    }

    private void removeExcludedPackages(Set<String> packageNames, List<String> exclusions) {
        for (String packageName : new LinkedHashSet<String>(packageNames)) {
            if (isExcluded(packageName, exclusions)) {
                packageNames.remove(packageName);
            }
        }
    }

    private boolean isExcluded(String exportedPackage, List<String> exclusions) {
        for (String exclusion : exclusions) {
            if (MatchUtils.matches(exportedPackage, exclusion)) {
                return true;
            }
        }
        return false;
    }

    public List<String> getTemplateOnlyHeaderNames() {
        return Arrays.asList(ATTR_EXCLUDED_EXPORTS, ATTR_EXCLUDED_IMPORTS);
    }
}
