/*
 * Copyright 2008-2009 SpringSource
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.springsource.bundlor.support.asm;

import org.objectweb.asm.AnnotationVisitor;
import org.objectweb.asm.Label;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.EmptyVisitor;

import com.springsource.bundlor.support.partialmanifest.PartialManifest;

/**
 * ASM {@link MethodVisitor} to scan method bodies for imports.
 * <p/>
 * 
 * <strong>Concurrent Semantics</strong><br />
 * 
 * Not threadsafe.
 * 
 * @author Rob Harrop
 * @author Christian Dupuis
 */
final class ArtefactAnalyserMethodVisitor extends EmptyVisitor implements MethodVisitor {

    /**
     * The <code>PartialManifest</code> being updated.
     */
    private final PartialManifest partialManifest;

    /**
     * The type that is being scanned.
     */
    private final Type type;

    /**
     * Creates a new <code>ArtefactAnalyserMethodVisitor</code> for the supplied {@link PartialManifest}.
     * 
     * @param partialManifest the <code>PartialManifest</code>.
     */
    public ArtefactAnalyserMethodVisitor(PartialManifest partialManifest, Type type) {
        this.partialManifest = partialManifest;
        this.type = type;
    }

    /**
     * @inheritDoc
     */
    public AnnotationVisitor visitAnnotation(String desc, boolean visible) {
        Type t = Type.getType(desc);
        VisitorUtils.recordReferencedTypes(partialManifest, t);
        VisitorUtils.recordUses(partialManifest, this.type, t);
        return null;
    }

    /**
     * @inheritDoc
     */
    public void visitFieldInsn(int opcode, String owner, String name, String desc) {
        VisitorUtils.recordReferencedTypes(partialManifest, Type.getType(desc));
        VisitorUtils.recordReferencedTypes(partialManifest, Type.getObjectType(owner));
    }

    /**
     * @inheritDoc
     */
    public void visitLocalVariable(String name, String desc, String signature, Label start, Label end, int index) {
        Type t = Type.getType(desc);
        VisitorUtils.recordReferencedTypes(partialManifest, t);
    }

    /**
     * @inheritDoc
     */
    public void visitMethodInsn(int opcode, String owner, String name, String desc) {
        Type t = Type.getObjectType(owner);
        VisitorUtils.recordReferencedTypes(partialManifest, t);
        VisitorUtils.recordReferencedTypes(partialManifest, Type.getReturnType(desc));
        VisitorUtils.recordReferencedTypes(partialManifest, Type.getArgumentTypes(desc));
    }

    /**
     * @inheritDoc
     */
    public void visitMultiANewArrayInsn(String desc, int dims) {
        Type t = Type.getType(desc);
        VisitorUtils.recordReferencedTypes(partialManifest, t);
    }

    /**
     * @inheritDoc
     */
    public AnnotationVisitor visitParameterAnnotation(int parameter, String desc, boolean visible) {
        Type t = Type.getType(desc);
        VisitorUtils.recordReferencedTypes(partialManifest, t);
        VisitorUtils.recordUses(partialManifest, this.type, t);
        return null;
    }

    /**
     * @inheritDoc
     */
    public void visitTryCatchBlock(Label start, Label end, Label handler, String type) {
        if (type != null) {
            Type t = Type.getObjectType(type);
            VisitorUtils.recordReferencedTypes(partialManifest, t);
        }
    }

    /**
     * @inheritDoc
     */
    public void visitTypeInsn(int opcode, String type) {
        Type t = Type.getObjectType(type);
        VisitorUtils.recordReferencedTypes(partialManifest, t);
    }

    /**
     * {@inheritDoc}
     */
    public void visitLdcInsn(Object cst) {
        if (cst instanceof Type) {
            VisitorUtils.recordReferencedTypes(partialManifest, (Type) cst);
        }
    }

}
