/*
 * Copyright 2009 SpringSource
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.springsource.bundlor.commandline.internal;

import java.util.List;

import com.springsource.bundlor.ClassPath;
import com.springsource.bundlor.ManifestGenerator;
import com.springsource.bundlor.ManifestWriter;
import com.springsource.bundlor.blint.ManifestValidator;
import com.springsource.bundlor.blint.support.DefaultManifestValidatorContributorsFactory;
import com.springsource.bundlor.blint.support.StandardManifestValidator;
import com.springsource.bundlor.commandline.BundlorExecutor;
import com.springsource.bundlor.support.DefaultManifestGeneratorContributorsFactory;
import com.springsource.bundlor.support.ManifestGeneratorContributors;
import com.springsource.bundlor.support.StandardManifestGenerator;
import com.springsource.bundlor.support.classpath.ClassPathFactory;
import com.springsource.bundlor.support.manifestwriter.ManifestWriterFactory;
import com.springsource.bundlor.support.properties.PropertiesSource;
import com.springsource.util.parser.manifest.ManifestContents;

public final class CommandLineBundlorExecutor implements BundlorExecutor {

    private final ClassPath inputClassPath;

    private final ManifestWriter manifestWriter;

    private final ManifestContents manifestTemplate;

    private final ManifestGenerator manifestGenerator;

    private final ManifestValidator manifestValidator;

    public CommandLineBundlorExecutor(Configuration configuration, ConfigurationValidator configurationValidator, ClassPathFactory classPathFactory,
        ManifestWriterFactory manifestWriterFactory, ManifestTemplateFactory manifestTemplateFactory,
        PropertiesSourceFactory propertiesSourceFactory, OsgiProfileFactory osgiProfileFactory) {

        configurationValidator.validate(configuration);

        this.inputClassPath = classPathFactory.create(configuration.getInputPath());
        this.manifestWriter = manifestWriterFactory.create(configuration.getInputPath(), configuration.getOutputPath());
        this.manifestTemplate = manifestTemplateFactory.create(configuration.getManifestTemplatePath());

        List<PropertiesSource> properties = propertiesSourceFactory.create(configuration.getPropertiesPath(), configuration.getProperties());
        properties.add(osgiProfileFactory.create(configuration.getOsgiProfilePath()));
        this.manifestGenerator = new StandardManifestGenerator(createContributors(properties));

        this.manifestValidator = new StandardManifestValidator(DefaultManifestValidatorContributorsFactory.create());
    }

    public List<String> execute() {
        ManifestContents manifest = this.manifestGenerator.generate(this.manifestTemplate, this.inputClassPath);

        try {
            this.manifestWriter.write(manifest);
        } finally {
            this.manifestWriter.close();
        }

        return this.manifestValidator.validate(manifest);
    }

    private ManifestGeneratorContributors createContributors(List<PropertiesSource> properties) {
        return DefaultManifestGeneratorContributorsFactory.create(properties.toArray(new PropertiesSource[properties.size()]));
    }

}
