/*
 * Copyright 2009 SpringSource
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.springsource.bundlor.commandline;

import java.util.List;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;

import com.springsource.bundlor.commandline.internal.CommandLineBundlorExecutor;
import com.springsource.bundlor.commandline.internal.Configuration;
import com.springsource.bundlor.commandline.internal.OptionsFactory;
import com.springsource.bundlor.commandline.internal.support.StandardConfigurationValidator;
import com.springsource.bundlor.commandline.internal.support.StandardManifestTemplateFactory;
import com.springsource.bundlor.commandline.internal.support.StandardOsgiProfileFactory;
import com.springsource.bundlor.commandline.internal.support.StandardPropertiesSourceFactory;
import com.springsource.bundlor.support.classpath.StandardClassPathFactory;
import com.springsource.bundlor.support.manifestwriter.StandardManifestWriterFactory;

public class Bundlor {

    private static final Options OPTIONS = new OptionsFactory().create();

    public final static void main(String[] args) {
        new Bundlor().run(args);
    }

    final void run(String[] args) {
        CommandLine commandLine = null;
        Configuration configuration = null;
        try {
            commandLine = new PosixParser().parse(OPTIONS, args);
            configuration = new Configuration(commandLine);
        } catch (ParseException e) {
            new HelpFormatter().printHelp("bundlor.[sh | bat] [OPTION]...", OPTIONS);
            exit(-1);
        }

        List<String> warnings = getBundlorExecutor(configuration).execute();

        if (warnings.size() > 0) {
            System.out.println("Bundlor Warnings:");
            for (String warning : warnings) {
                System.out.println("    " + warning);
            }

            if (commandLine.hasOption("f")) {
                String message = String.format("Bundlor returned warnings.  Please fix manifest template at '%s' and try again.",
                    configuration.getManifestTemplatePath());
                System.err.println(message);
                exit(-1);
            }
        }
    }

    protected BundlorExecutor getBundlorExecutor(Configuration configuration) {
        return new CommandLineBundlorExecutor(configuration, new StandardConfigurationValidator(), new StandardClassPathFactory(),
            new StandardManifestWriterFactory(), new StandardManifestTemplateFactory(), new StandardPropertiesSourceFactory(),
            new StandardOsgiProfileFactory());
    }

    protected void exit(int code) {
        System.exit(code);
    }

}
