/*
 * Copyright 2009 SpringSource
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.springsource.bundlor.blint.support.contributors;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.springsource.bundlor.blint.support.Validator;
import com.springsource.bundlor.util.BundleManifestUtils;
import com.springsource.util.osgi.VersionRange;
import com.springsource.util.osgi.manifest.BundleManifest;
import com.springsource.util.osgi.manifest.ImportedPackage;
import com.springsource.util.parser.manifest.ManifestContents;

public final class VersionedImportsRangeValidator implements Validator {

    private static final String MESSAGE = "The import of package %s has an unlikely version";

    public Set<String> validate(ManifestContents manifest) {
        Set<String> warnings = new HashSet<String>();

        BundleManifest bundleManifest = BundleManifestUtils.createBundleManifest(manifest);
        List<ImportedPackage> importedPackages = bundleManifest.getImportPackage().getImportedPackages();
        for (ImportedPackage importedPackage : importedPackages) {
            try {
                VersionRange versionRange = importedPackage.getVersion();

                if (isReversed(versionRange) || isExclusive(versionRange)) {
                    warnings.add(String.format(MESSAGE, importedPackage.getPackageName()));
                }
            } catch (IllegalArgumentException e) {
                warnings.add(String.format(MESSAGE, importedPackage.getPackageName()));
            }
        }

        return warnings;
    }

    private boolean isReversed(VersionRange versionRange) {
        if (versionRange.getCeiling() == null) {
            return false;
        }
        return versionRange.getFloor().compareTo(versionRange.getCeiling()) > 0;
    }

    private boolean isExclusive(VersionRange versionRange) {
        if (versionRange.getFloor().equals(versionRange.getCeiling())) {
            return !versionRange.isFloorInclusive() && !versionRange.isCeilingInclusive();
        }
        return false;
    }
}
