/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.servo.monitor;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.servo.monitor.AbstractMonitor;
import com.netflix.servo.monitor.BasicCounter;
import com.netflix.servo.monitor.CompositeMonitor;
import com.netflix.servo.monitor.Counter;
import com.netflix.servo.monitor.DoubleGauge;
import com.netflix.servo.monitor.LongGauge;
import com.netflix.servo.monitor.Monitor;
import com.netflix.servo.monitor.MonitorConfig;
import com.netflix.servo.monitor.NumericMonitor;
import com.netflix.servo.stats.StatsBuffer;
import com.netflix.servo.stats.StatsConfig;
import com.netflix.servo.tag.BasicTagList;
import com.netflix.servo.tag.Tag;
import com.netflix.servo.tag.Tags;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StatsMonitor
extends AbstractMonitor<Long>
implements CompositeMonitor<Long>,
NumericMonitor<Long> {
    private static final ThreadFactory threadFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("StatsMonitor-%d").build();
    protected static final ScheduledExecutorService defaultExecutor = Executors.newScheduledThreadPool(1, threadFactory);
    private static final Logger LOGGER = LoggerFactory.getLogger(StatsMonitor.class);
    private final MonitorConfig baseConfig;
    protected final Counter count;
    protected final Counter totalMeasurement;
    private final List<Monitor<?>> monitors;
    private final List<GaugeWrapper> gaugeWrappers;
    private final Runnable startComputingAction;
    private final Object updateLock = new Object();
    private StatsBuffer cur;
    private StatsBuffer prev;
    private static final String STATISTIC = "statistic";
    private static final String PERCENTILE_FMT = "percentile_%.2f";
    private static final Tag STAT_COUNT = Tags.newTag("statistic", "count");
    private static final Tag STAT_MIN = Tags.newTag("statistic", "min");
    private static final Tag STAT_MAX = Tags.newTag("statistic", "max");
    private static final Tag STAT_MEAN = Tags.newTag("statistic", "avg");
    private static final Tag STAT_VARIANCE = Tags.newTag("statistic", "variance");
    private static final Tag STAT_STDDEV = Tags.newTag("statistic", "stdDev");

    private List<Counter> getCounters(StatsConfig config) {
        ImmutableList.Builder monitors = ImmutableList.builder();
        if (config.getPublishCount()) {
            monitors.add((Object)this.count);
        }
        if (config.getPublishTotal()) {
            monitors.add((Object)this.totalMeasurement);
        }
        return monitors.build();
    }

    private List<GaugeWrapper> getGaugeWrappers(StatsConfig config) {
        ImmutableList.Builder builder = ImmutableList.builder();
        if (config.getPublishMax()) {
            builder.add((Object)new MaxGaugeWrapper(this.baseConfig));
        }
        if (config.getPublishMin()) {
            builder.add((Object)new MinStatGaugeWrapper(this.baseConfig));
        }
        if (config.getPublishVariance()) {
            builder.add((Object)new VarianceGaugeWrapper(this.baseConfig));
        }
        if (config.getPublishStdDev()) {
            builder.add((Object)new StdDevGaugeWrapper(this.baseConfig));
        }
        if (config.getPublishMean()) {
            builder.add((Object)new MeanGaugeWrapper(this.baseConfig));
        }
        double[] percentiles = config.getPercentiles();
        for (int i = 0; i < percentiles.length; ++i) {
            builder.add((Object)new PercentileGaugeWrapper(this.baseConfig, percentiles[i], i));
        }
        ImmutableList wrappers = builder.build();
        HashSet seen = Sets.newHashSet();
        for (GaugeWrapper wrapper : wrappers) {
            MonitorConfig cfg = wrapper.getMonitor().getConfig();
            if (seen.contains(cfg)) {
                throw new IllegalArgumentException("Duplicated monitor configuration found: " + cfg);
            }
            seen.add(cfg);
        }
        return wrappers;
    }

    public StatsMonitor(MonitorConfig config, final StatsConfig statsConfig, final ScheduledExecutorService executor, String totalTagName, boolean autoStart, Tag ... additionalTags) {
        super(config);
        Tag statsTotal = Tags.newTag(STATISTIC, totalTagName);
        BasicTagList additionalTagList = new BasicTagList(Arrays.asList(additionalTags));
        this.baseConfig = config.withAdditionalTags(additionalTagList);
        this.cur = new StatsBuffer(statsConfig.getSampleSize(), statsConfig.getPercentiles());
        this.prev = new StatsBuffer(statsConfig.getSampleSize(), statsConfig.getPercentiles());
        this.count = new BasicCounter(this.baseConfig.withAdditionalTag(STAT_COUNT));
        this.totalMeasurement = new BasicCounter(this.baseConfig.withAdditionalTag(statsTotal));
        this.gaugeWrappers = this.getGaugeWrappers(statsConfig);
        Collection gaugeMonitors = Collections2.transform(this.gaugeWrappers, (Function)new Function<GaugeWrapper, Monitor<?>>(){

            public Monitor<?> apply(GaugeWrapper perfStatGauge) {
                return perfStatGauge.getMonitor();
            }
        });
        this.monitors = new ImmutableList.Builder().addAll(this.getCounters(statsConfig)).addAll((Iterable)gaugeMonitors).build();
        this.startComputingAction = new Runnable(){

            @Override
            public void run() {
                StatsMonitor.this.startComputingStats(executor, statsConfig.getFrequencyMillis());
            }
        };
        if (autoStart) {
            this.startComputingStats();
        }
    }

    public void startComputingStats() {
        this.startComputingAction.run();
    }

    private void startComputingStats(ScheduledExecutorService executor, long frequencyMillis) {
        Runnable command = new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                try {
                    Object object = StatsMonitor.this.updateLock;
                    synchronized (object) {
                        StatsBuffer tmp = StatsMonitor.this.prev;
                        StatsMonitor.this.prev = StatsMonitor.this.cur;
                        StatsMonitor.this.cur = tmp;
                    }
                    StatsMonitor.this.prev.computeStats();
                    StatsMonitor.this.updateGauges();
                    StatsMonitor.this.prev.reset();
                }
                catch (Exception e) {
                    StatsMonitor.this.handleException(e);
                }
            }
        };
        executor.scheduleWithFixedDelay(command, frequencyMillis, frequencyMillis, TimeUnit.MILLISECONDS);
    }

    private void updateGauges() {
        for (GaugeWrapper gauge : this.gaugeWrappers) {
            gauge.update(this.prev);
        }
    }

    @Override
    public List<Monitor<?>> getMonitors() {
        return this.monitors;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void record(long measurement) {
        Object object = this.updateLock;
        synchronized (object) {
            this.cur.record(measurement);
        }
        this.count.increment();
        this.totalMeasurement.increment(measurement);
    }

    @Override
    public Long getValue() {
        long n = this.getCount();
        return n > 0L ? ((Number)this.totalMeasurement.getValue()).longValue() / n : 0L;
    }

    protected void handleException(Exception e) {
        LOGGER.warn("Unable to compute stats: ", (Throwable)e);
    }

    public String toString() {
        return Objects.toStringHelper((Object)this).add("baseConfig", (Object)this.baseConfig).add("monitors", this.monitors).toString();
    }

    public boolean equals(Object obj) {
        if (obj == null || !(obj instanceof StatsMonitor)) {
            return false;
        }
        StatsMonitor m = (StatsMonitor)obj;
        return this.baseConfig.equals(m.baseConfig) && ((Object)this.monitors).equals(m.monitors);
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.baseConfig, this.monitors});
    }

    public long getCount() {
        return ((Number)this.count.getValue()).longValue();
    }

    public long getTotalMeasurement() {
        return ((Number)this.totalMeasurement.getValue()).longValue();
    }

    private static class PercentileGaugeWrapper
    extends DoubleGaugeWrapper {
        private final double percentile;
        private final int index;

        private static Tag percentileTag(double percentile) {
            String percentileStr = String.format(StatsMonitor.PERCENTILE_FMT, percentile);
            if (percentileStr.endsWith(".00")) {
                percentileStr = percentileStr.substring(0, percentileStr.length() - 3);
            }
            return Tags.newTag(StatsMonitor.STATISTIC, percentileStr);
        }

        PercentileGaugeWrapper(MonitorConfig baseConfig, double percentile, int index) {
            super(baseConfig.withAdditionalTag(PercentileGaugeWrapper.percentileTag(percentile)));
            this.percentile = percentile;
            this.index = index;
        }

        @Override
        public void update(StatsBuffer buffer) {
            this.gauge.set(buffer.getPercentileValues()[this.index]);
        }

        @Override
        public String toString() {
            return Objects.toStringHelper((Object)this).add("gauge", (Object)this.gauge).add("percentile", this.percentile).toString();
        }
    }

    private static class StdDevGaugeWrapper
    extends DoubleGaugeWrapper {
        StdDevGaugeWrapper(MonitorConfig baseConfig) {
            super(baseConfig.withAdditionalTag(STAT_STDDEV));
        }

        @Override
        public void update(StatsBuffer buffer) {
            this.gauge.set(buffer.getStdDev());
        }
    }

    private static class VarianceGaugeWrapper
    extends DoubleGaugeWrapper {
        VarianceGaugeWrapper(MonitorConfig baseConfig) {
            super(baseConfig.withAdditionalTag(STAT_VARIANCE));
        }

        @Override
        public void update(StatsBuffer buffer) {
            this.gauge.set(buffer.getVariance());
        }
    }

    private static class MeanGaugeWrapper
    extends DoubleGaugeWrapper {
        MeanGaugeWrapper(MonitorConfig baseConfig) {
            super(baseConfig.withAdditionalTag(STAT_MEAN));
        }

        @Override
        public void update(StatsBuffer buffer) {
            this.gauge.set(buffer.getMean());
        }
    }

    private static class MaxGaugeWrapper
    extends LongGaugeWrapper {
        MaxGaugeWrapper(MonitorConfig baseConfig) {
            super(baseConfig.withAdditionalTag(STAT_MAX));
        }

        @Override
        public void update(StatsBuffer buffer) {
            this.gauge.set(buffer.getMax());
        }
    }

    private static class MinStatGaugeWrapper
    extends LongGaugeWrapper {
        MinStatGaugeWrapper(MonitorConfig baseConfig) {
            super(baseConfig.withAdditionalTag(STAT_MIN));
        }

        @Override
        public void update(StatsBuffer buffer) {
            this.gauge.set(buffer.getMin());
        }
    }

    private static abstract class DoubleGaugeWrapper
    implements GaugeWrapper {
        protected final DoubleGauge gauge;

        protected DoubleGaugeWrapper(MonitorConfig config) {
            this.gauge = new DoubleGauge(config);
        }

        @Override
        public Monitor<?> getMonitor() {
            return this.gauge;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof DoubleGaugeWrapper)) {
                return false;
            }
            DoubleGaugeWrapper that = (DoubleGaugeWrapper)o;
            return this.gauge.equals(that.gauge);
        }

        public int hashCode() {
            return this.gauge.hashCode();
        }

        public String toString() {
            return Objects.toStringHelper((Object)this).add("gauge", (Object)this.gauge).toString();
        }
    }

    private static abstract class LongGaugeWrapper
    implements GaugeWrapper {
        protected final LongGauge gauge;

        protected LongGaugeWrapper(MonitorConfig config) {
            this.gauge = new LongGauge(config);
        }

        @Override
        public Monitor<?> getMonitor() {
            return this.gauge;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof LongGaugeWrapper)) {
                return false;
            }
            LongGaugeWrapper that = (LongGaugeWrapper)o;
            return this.gauge.equals(that.gauge);
        }

        public int hashCode() {
            return this.gauge.hashCode();
        }

        public String toString() {
            return Objects.toStringHelper((Object)this).add("gauge", (Object)this.gauge).toString();
        }
    }

    private static interface GaugeWrapper {
        public void update(StatsBuffer var1);

        public Monitor<?> getMonitor();
    }
}

