/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.curator;

import com.google.common.io.Closeables;
import com.netflix.curator.HandleHolder;
import com.netflix.curator.SessionFailRetryLoop;
import com.netflix.curator.TimeTrace;
import com.netflix.curator.drivers.TracerDriver;
import com.netflix.curator.ensemble.EnsembleProvider;
import com.netflix.curator.utils.ZookeeperFactory;
import java.io.Closeable;
import java.io.IOException;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConnectionState
implements Watcher,
Closeable {
    private volatile long connectionStartMs = 0L;
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final HandleHolder zooKeeper;
    private final AtomicBoolean isConnected = new AtomicBoolean(false);
    private final EnsembleProvider ensembleProvider;
    private final int connectionTimeoutMs;
    private final AtomicReference<TracerDriver> tracer;
    private final Queue<Exception> backgroundExceptions = new ConcurrentLinkedQueue<Exception>();
    private final Queue<Watcher> parentWatchers = new ConcurrentLinkedQueue<Watcher>();
    private static final int MAX_BACKGROUND_EXCEPTIONS = 10;
    private static final boolean LOG_EVENTS = Boolean.getBoolean("curator-log-events");

    ConnectionState(ZookeeperFactory zookeeperFactory, EnsembleProvider ensembleProvider, int n, int n2, Watcher watcher, AtomicReference<TracerDriver> atomicReference) throws IOException {
        this.ensembleProvider = ensembleProvider;
        this.connectionTimeoutMs = n2;
        this.tracer = atomicReference;
        if (watcher != null) {
            this.parentWatchers.offer(watcher);
        }
        this.zooKeeper = new HandleHolder(zookeeperFactory, this, ensembleProvider, n);
    }

    ZooKeeper getZooKeeper() throws Exception {
        long l;
        if (SessionFailRetryLoop.sessionForThreadHasFailed()) {
            throw new SessionFailRetryLoop.SessionFailedException();
        }
        Exception exception = this.backgroundExceptions.poll();
        if (exception != null) {
            this.log.error("Background exception caught", (Throwable)exception);
            this.tracer.get().addCount("background-exceptions", 1);
            throw exception;
        }
        boolean bl = this.isConnected.get();
        if (!bl && (l = System.currentTimeMillis() - this.connectionStartMs) >= (long)this.connectionTimeoutMs) {
            if (this.zooKeeper.hasNewConnectionString()) {
                this.handleNewConnectionString();
            } else {
                KeeperException.ConnectionLossException connectionLossException = new KeeperException.ConnectionLossException();
                if (!Boolean.getBoolean("curator-dont-log-connection-problems")) {
                    this.log.error(String.format("Connection timed out for connection string (%s) and timeout (%d) / elapsed (%d)", this.zooKeeper.getConnectionString(), this.connectionTimeoutMs, l), (Throwable)connectionLossException);
                }
                this.tracer.get().addCount("connections-timed-out", 1);
                throw connectionLossException;
            }
        }
        return this.zooKeeper.getZooKeeper();
    }

    boolean isConnected() {
        return this.isConnected.get();
    }

    void start() throws Exception {
        this.log.debug("Starting");
        this.ensembleProvider.start();
        this.reset();
    }

    @Override
    public void close() throws IOException {
        this.log.debug("Closing");
        Closeables.closeQuietly((Closeable)this.ensembleProvider);
        try {
            this.zooKeeper.closeAndClear();
        }
        catch (Exception exception) {
            throw new IOException(exception);
        }
        finally {
            this.isConnected.set(false);
        }
    }

    void addParentWatcher(Watcher watcher) {
        this.parentWatchers.offer(watcher);
    }

    void removeParentWatcher(Watcher watcher) {
        this.parentWatchers.remove(watcher);
    }

    private void reset() throws Exception {
        this.isConnected.set(false);
        this.connectionStartMs = System.currentTimeMillis();
        this.zooKeeper.closeAndReset();
        this.zooKeeper.getZooKeeper();
    }

    public void process(WatchedEvent watchedEvent) {
        boolean bl;
        if (LOG_EVENTS) {
            this.log.debug("ConnectState watcher: " + watchedEvent);
        }
        for (Watcher watcher : this.parentWatchers) {
            TimeTrace timeTrace = new TimeTrace("connection-state-parent-process", this.tracer.get());
            watcher.process(watchedEvent);
            timeTrace.commit();
        }
        boolean bl2 = bl = this.isConnected.get();
        if (watchedEvent.getType() == Watcher.Event.EventType.None) {
            bl2 = this.checkState(watchedEvent.getState(), bl);
        }
        if (bl2 != bl) {
            this.isConnected.set(bl2);
            this.connectionStartMs = System.currentTimeMillis();
        }
    }

    private boolean checkState(Watcher.Event.KeeperState keeperState, boolean bl) {
        boolean bl2 = bl;
        boolean bl3 = true;
        switch (keeperState) {
            default: {
                bl2 = false;
                break;
            }
            case SyncConnected: 
            case ConnectedReadOnly: {
                bl2 = true;
                break;
            }
            case AuthFailed: {
                bl2 = false;
                this.log.error("Authentication failed");
                break;
            }
            case Expired: {
                bl2 = false;
                bl3 = false;
                this.handleExpiredSession();
            }
            case SaslAuthenticated: 
        }
        if (bl3 && this.zooKeeper.hasNewConnectionString()) {
            this.handleNewConnectionString();
        }
        return bl2;
    }

    private void handleNewConnectionString() {
        this.log.info("Connection string changed");
        this.tracer.get().addCount("connection-string-changed", 1);
        try {
            this.reset();
        }
        catch (Exception exception) {
            this.queueBackgroundException(exception);
        }
    }

    private void handleExpiredSession() {
        this.log.warn("Session expired event received");
        this.tracer.get().addCount("session-expired", 1);
        try {
            this.reset();
        }
        catch (Exception exception) {
            this.queueBackgroundException(exception);
        }
    }

    private void queueBackgroundException(Exception exception) {
        while (this.backgroundExceptions.size() >= 10) {
            this.backgroundExceptions.poll();
        }
        this.backgroundExceptions.offer(exception);
    }
}

