/*
 * Decompiled with CFR 0.152.
 */
package com.google.devtools.common.options;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.devtools.common.options.Converter;
import com.google.devtools.common.options.EnumConverter;
import com.google.devtools.common.options.OptionsParser;
import com.google.devtools.common.options.OptionsParsingException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

public final class Converters {
    static String joinEnglishList(Iterable<?> choices) {
        StringBuilder buf = new StringBuilder();
        Iterator<?> ii = choices.iterator();
        while (ii.hasNext()) {
            Object choice = ii.next();
            if (buf.length() > 0) {
                buf.append(ii.hasNext() ? ", " : " or ");
            }
            buf.append(choice);
        }
        return buf.length() == 0 ? "nothing" : buf.toString();
    }

    public static class BooleanConverter
    implements Converter<Boolean> {
        @Override
        public Boolean convert(String input) throws OptionsParsingException {
            if (input == null) {
                return false;
            }
            if ((input = input.toLowerCase()).equals("true") || input.equals("1") || input.equals("yes") || input.equals("t") || input.equals("y")) {
                return true;
            }
            if (input.equals("false") || input.equals("0") || input.equals("no") || input.equals("f") || input.equals("n")) {
                return false;
            }
            throw new OptionsParsingException("'" + input + "' is not a boolean");
        }

        @Override
        public String getTypeDescription() {
            return "a boolean";
        }
    }

    public static class HelpVerbosityConverter
    extends EnumConverter<OptionsParser.HelpVerbosity> {
        public HelpVerbosityConverter() {
            super(OptionsParser.HelpVerbosity.class, "--help_verbosity setting");
        }
    }

    public static class OptionalAssignmentConverter
    implements Converter<Map.Entry<String, String>> {
        @Override
        public Map.Entry<String, String> convert(String input) throws OptionsParsingException {
            int pos = input.indexOf("=");
            if (pos == 0 || input.length() == 0) {
                throw new OptionsParsingException("Variable definitions must be in the form of a 'name=value' or 'name' assignment");
            }
            if (pos < 0) {
                return Maps.immutableEntry((Object)input, null);
            }
            String name = input.substring(0, pos);
            String value = input.substring(pos + 1);
            return Maps.immutableEntry((Object)name, (Object)value);
        }

        @Override
        public String getTypeDescription() {
            return "a 'name=value' assignment with an optional value part";
        }
    }

    public static class AssignmentConverter
    implements Converter<Map.Entry<String, String>> {
        @Override
        public Map.Entry<String, String> convert(String input) throws OptionsParsingException {
            int pos = input.indexOf("=");
            if (pos <= 0) {
                throw new OptionsParsingException("Variable definitions must be in the form of a 'name=value' assignment");
            }
            String name = input.substring(0, pos);
            String value = input.substring(pos + 1);
            return Maps.immutableEntry((Object)name, (Object)value);
        }

        @Override
        public String getTypeDescription() {
            return "a 'name=value' assignment";
        }
    }

    public static class RangeConverter
    implements Converter<Integer> {
        final int minValue;
        final int maxValue;

        public RangeConverter(int minValue, int maxValue) {
            this.minValue = minValue;
            this.maxValue = maxValue;
        }

        @Override
        public Integer convert(String input) throws OptionsParsingException {
            try {
                Integer value = Integer.parseInt(input);
                if (value < this.minValue) {
                    throw new OptionsParsingException("'" + input + "' should be >= " + this.minValue);
                }
                if (value < this.minValue || value > this.maxValue) {
                    throw new OptionsParsingException("'" + input + "' should be <= " + this.maxValue);
                }
                return value;
            }
            catch (NumberFormatException e) {
                throw new OptionsParsingException("'" + input + "' is not an int");
            }
        }

        @Override
        public String getTypeDescription() {
            if (this.minValue == Integer.MIN_VALUE) {
                if (this.maxValue == Integer.MAX_VALUE) {
                    return "an integer";
                }
                return "an integer, <= " + this.maxValue;
            }
            if (this.maxValue == Integer.MAX_VALUE) {
                return "an integer, >= " + this.minValue;
            }
            return "an integer in " + (this.minValue < 0 ? "(" + this.minValue + ")" : Integer.valueOf(this.minValue)) + "-" + this.maxValue + " range";
        }
    }

    public static class LengthLimitingConverter
    implements Converter<String> {
        private final int maxSize;

        public LengthLimitingConverter(int maxSize) {
            this.maxSize = maxSize;
        }

        @Override
        public String convert(String input) throws OptionsParsingException {
            if (input.length() > this.maxSize) {
                throw new OptionsParsingException("Input must be " + this.getTypeDescription());
            }
            return input;
        }

        @Override
        public String getTypeDescription() {
            return "a string <= " + this.maxSize + " characters";
        }
    }

    public static class RegexPatternConverter
    implements Converter<Pattern> {
        @Override
        public Pattern convert(String input) throws OptionsParsingException {
            try {
                return Pattern.compile(input);
            }
            catch (PatternSyntaxException e) {
                throw new OptionsParsingException("Not a valid regular expression: " + e.getMessage());
            }
        }

        @Override
        public String getTypeDescription() {
            return "a valid Java regular expression";
        }
    }

    public static class StringSetConverter
    implements Converter<String> {
        private final List<String> values;

        public StringSetConverter(String ... values) {
            this.values = ImmutableList.copyOf((Object[])values);
        }

        @Override
        public String convert(String input) throws OptionsParsingException {
            if (this.values.contains(input)) {
                return input;
            }
            throw new OptionsParsingException("Not one of " + this.values);
        }

        @Override
        public String getTypeDescription() {
            return Converters.joinEnglishList(this.values);
        }
    }

    public static class LogLevelConverter
    implements Converter<Level> {
        public static Level[] LEVELS = new Level[]{Level.OFF, Level.SEVERE, Level.WARNING, Level.INFO, Level.FINE, Level.FINER, Level.FINEST};

        @Override
        public Level convert(String input) throws OptionsParsingException {
            try {
                int level = Integer.parseInt(input);
                return LEVELS[level];
            }
            catch (NumberFormatException e) {
                throw new OptionsParsingException("Not a log level: " + input);
            }
            catch (ArrayIndexOutOfBoundsException e) {
                throw new OptionsParsingException("Not a log level: " + input);
            }
        }

        @Override
        public String getTypeDescription() {
            return "0 <= an integer <= " + (LEVELS.length - 1);
        }
    }

    public static class ColonSeparatedOptionListConverter
    extends SeparatedOptionListConverter {
        public ColonSeparatedOptionListConverter() {
            super(':', "colon");
        }
    }

    public static class CommaSeparatedOptionListConverter
    extends SeparatedOptionListConverter {
        public CommaSeparatedOptionListConverter() {
            super(',', "comma");
        }
    }

    public static class SeparatedOptionListConverter
    implements Converter<List<String>> {
        private final String separatorDescription;
        private final Splitter splitter;

        protected SeparatedOptionListConverter(char separator, String separatorDescription) {
            this.separatorDescription = separatorDescription;
            this.splitter = Splitter.on((char)separator);
        }

        @Override
        public List<String> convert(String input) {
            return input.equals("") ? ImmutableList.of() : ImmutableList.copyOf((Iterable)this.splitter.split((CharSequence)input));
        }

        @Override
        public String getTypeDescription() {
            return this.separatorDescription + "-separated list of options";
        }
    }
}

