/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.implementation;

import com.azure.core.credential.AccessToken;
import com.azure.core.credential.TokenCredential;
import com.azure.core.credential.TokenRequestContext;
import com.azure.core.implementation.AccessTokenCacheInfo;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LogLevel;
import com.azure.core.util.logging.LoggingEventBuilder;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Signal;
import reactor.core.publisher.Sinks;

public final class AccessTokenCache {
    private static final Duration REFRESH_DELAY = Duration.ofSeconds(30L);
    private static final String REFRESH_DELAY_STRING = String.valueOf(REFRESH_DELAY.getSeconds());
    private static final Duration REFRESH_OFFSET = Duration.ofMinutes(5L);
    private static final ClientLogger LOGGER = new ClientLogger(AccessTokenCache.class);
    private final AtomicReference<Sinks.One<AccessToken>> wip;
    private final AtomicReference<AccessTokenCacheInfo> cacheInfo;
    private final TokenCredential tokenCredential;
    private TokenRequestContext tokenRequestContext;
    private final Supplier<Mono<AccessToken>> tokenSupplierAsync;
    private final Supplier<AccessToken> tokenSupplierSync;
    private final Predicate<AccessToken> shouldRefresh;
    private final Lock lock;

    public AccessTokenCache(TokenCredential tokenCredential) {
        Objects.requireNonNull(tokenCredential, "The token credential cannot be null");
        this.wip = new AtomicReference();
        this.tokenCredential = tokenCredential;
        this.cacheInfo = new AtomicReference<AccessTokenCacheInfo>(new AccessTokenCacheInfo(null, OffsetDateTime.now()));
        this.shouldRefresh = accessToken -> OffsetDateTime.now().isAfter(accessToken.getExpiresAt().minus(REFRESH_OFFSET));
        this.tokenSupplierAsync = () -> tokenCredential.getToken(this.tokenRequestContext);
        this.tokenSupplierSync = () -> tokenCredential.getTokenSync(this.tokenRequestContext);
        this.lock = new ReentrantLock();
    }

    public Mono<AccessToken> getToken(TokenRequestContext tokenRequestContext, boolean checkToForceFetchToken) {
        return Mono.defer(this.retrieveToken(tokenRequestContext, checkToForceFetchToken)).repeatWhenEmpty(longFlux -> longFlux.concatMap(ignored -> Flux.just((Object)true).delayElements(Duration.ofMillis(500L))));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AccessToken getTokenSync(TokenRequestContext tokenRequestContext, boolean checkToForceFetchToken) {
        this.lock.lock();
        try {
            AccessToken accessToken = this.retrieveTokenSync(tokenRequestContext, checkToForceFetchToken).get();
            return accessToken;
        }
        finally {
            this.lock.unlock();
        }
    }

    private Supplier<Mono<? extends AccessToken>> retrieveToken(TokenRequestContext tokenRequestContext, boolean checkToForceFetchToken) {
        return () -> {
            try {
                if (tokenRequestContext == null) {
                    return Mono.error((Throwable)LOGGER.logExceptionAsError(new IllegalArgumentException("The token request context input cannot be null.")));
                }
                AccessTokenCacheInfo cache = this.cacheInfo.get();
                AccessToken cachedToken = cache.getCachedAccessToken();
                if (this.wip.compareAndSet(null, (Sinks.One<AccessToken>)Sinks.one())) {
                    Mono fallback;
                    Mono tokenRefresh;
                    boolean forceRefresh;
                    Sinks.One<AccessToken> sinksOne = this.wip.get();
                    OffsetDateTime now = OffsetDateTime.now();
                    boolean bl = forceRefresh = checkToForceFetchToken && this.checkIfForceRefreshRequired(tokenRequestContext) || this.tokenRequestContext == null;
                    if (forceRefresh) {
                        this.tokenRequestContext = tokenRequestContext;
                        tokenRefresh = Mono.defer(() -> this.tokenCredential.getToken(this.tokenRequestContext));
                        fallback = Mono.empty();
                    } else if (cachedToken != null && !this.shouldRefresh.test(cachedToken)) {
                        tokenRefresh = Mono.empty();
                        fallback = Mono.just((Object)cachedToken);
                    } else if (cachedToken == null || cachedToken.isExpired()) {
                        tokenRefresh = Mono.defer(this.tokenSupplierAsync);
                        fallback = Mono.empty();
                    } else {
                        tokenRefresh = now.isAfter(cache.getNextTokenRefresh()) ? Mono.defer(this.tokenSupplierAsync) : Mono.empty();
                        fallback = Mono.just((Object)cachedToken);
                    }
                    return Mono.using(() -> this.wip, ignored -> tokenRefresh.materialize().flatMap(this.processTokenRefreshResult(sinksOne, now, (Mono<AccessToken>)fallback)).doOnError(arg_0 -> ((Sinks.One)sinksOne).tryEmitError(arg_0)), w -> w.set(null));
                }
                if (cachedToken != null && !cachedToken.isExpired() && !checkToForceFetchToken) {
                    return Mono.just((Object)cachedToken);
                }
                if (checkToForceFetchToken) {
                    return Mono.empty();
                }
                Sinks.One<AccessToken> sinksOne = this.wip.get();
                if (sinksOne == null) {
                    return Mono.just((Object)cachedToken);
                }
                return sinksOne.asMono().switchIfEmpty(Mono.fromSupplier(() -> cachedToken));
            }
            catch (Exception ex) {
                return Mono.error((Throwable)ex);
            }
        };
    }

    private Supplier<AccessToken> retrieveTokenSync(TokenRequestContext tokenRequestContext, boolean checkToForceFetchToken) {
        return () -> {
            AccessToken fallback;
            Supplier<AccessToken> tokenRefresh;
            boolean forceRefresh;
            if (tokenRequestContext == null) {
                throw LOGGER.logExceptionAsError(new IllegalArgumentException("The token request context input cannot be null."));
            }
            AccessTokenCacheInfo cache = this.cacheInfo.get();
            AccessToken cachedToken = cache.getCachedAccessToken();
            OffsetDateTime now = OffsetDateTime.now();
            boolean bl = forceRefresh = checkToForceFetchToken && this.checkIfForceRefreshRequired(tokenRequestContext) || this.tokenRequestContext == null;
            if (forceRefresh) {
                this.tokenRequestContext = tokenRequestContext;
                tokenRefresh = this.tokenSupplierSync;
                fallback = null;
            } else if (cachedToken != null && !this.shouldRefresh.test(cachedToken)) {
                tokenRefresh = null;
                fallback = cachedToken;
            } else if (cachedToken == null || cachedToken.isExpired()) {
                tokenRefresh = this.tokenSupplierSync;
                fallback = null;
            } else {
                tokenRefresh = now.isAfter(cache.getNextTokenRefresh()) ? this.tokenSupplierSync : null;
                fallback = cachedToken;
            }
            try {
                if (tokenRefresh != null) {
                    AccessToken token = tokenRefresh.get();
                    AccessTokenCache.buildTokenRefreshLog(LogLevel.INFORMATIONAL, cachedToken, now).log("Acquired a new access token.");
                    OffsetDateTime nextTokenRefreshTime = OffsetDateTime.now().plus(REFRESH_DELAY);
                    AccessTokenCacheInfo updatedInfo = new AccessTokenCacheInfo(token, nextTokenRefreshTime);
                    this.cacheInfo.set(updatedInfo);
                    return token;
                }
                return fallback;
            }
            catch (Throwable error) {
                AccessTokenCache.buildTokenRefreshLog(LogLevel.ERROR, cachedToken, now).log("Failed to acquire a new access token.", error);
                OffsetDateTime nextTokenRefreshTime = OffsetDateTime.now();
                AccessTokenCacheInfo updatedInfo = new AccessTokenCacheInfo(cachedToken, nextTokenRefreshTime);
                this.cacheInfo.set(updatedInfo);
                if (fallback != null) {
                    return fallback;
                }
                throw error;
            }
        };
    }

    private boolean checkIfForceRefreshRequired(TokenRequestContext tokenRequestContext) {
        return this.tokenRequestContext == null || !(this.tokenRequestContext.getClaims() == null ? tokenRequestContext.getClaims() == null : tokenRequestContext.getClaims() != null && tokenRequestContext.getClaims().equals(this.tokenRequestContext.getClaims())) || !this.tokenRequestContext.getScopes().equals(tokenRequestContext.getScopes());
    }

    private Function<Signal<AccessToken>, Mono<? extends AccessToken>> processTokenRefreshResult(Sinks.One<AccessToken> sinksOne, OffsetDateTime now, Mono<AccessToken> fallback) {
        return signal -> {
            AccessToken accessToken = (AccessToken)signal.get();
            Throwable error = signal.getThrowable();
            AccessToken cache = this.cacheInfo.get().getCachedAccessToken();
            if (signal.isOnNext() && accessToken != null) {
                AccessTokenCache.buildTokenRefreshLog(LogLevel.INFORMATIONAL, cache, now).log("Acquired a new access token.");
                sinksOne.tryEmitValue((Object)accessToken);
                OffsetDateTime nextTokenRefresh = OffsetDateTime.now().plus(REFRESH_DELAY);
                this.cacheInfo.set(new AccessTokenCacheInfo(accessToken, nextTokenRefresh));
                return Mono.just((Object)accessToken);
            }
            if (signal.isOnError() && error != null) {
                AccessTokenCache.buildTokenRefreshLog(LogLevel.ERROR, cache, now).log("Failed to acquire a new access token.", error);
                OffsetDateTime nextTokenRefresh = OffsetDateTime.now();
                this.cacheInfo.set(new AccessTokenCacheInfo(cache, nextTokenRefresh));
                return fallback.switchIfEmpty(Mono.error((Throwable)error));
            }
            sinksOne.tryEmitEmpty();
            return fallback;
        };
    }

    private static LoggingEventBuilder buildTokenRefreshLog(LogLevel level, AccessToken cache, OffsetDateTime now) {
        LoggingEventBuilder logBuilder = LOGGER.atLevel(level);
        if (cache == null || !LOGGER.canLogAtLevel(level)) {
            return logBuilder;
        }
        Duration tte = Duration.between(now, cache.getExpiresAt());
        return logBuilder.addKeyValue("expiresAt", cache.getExpiresAt()).addKeyValue("tteSeconds", String.valueOf(tte.abs().getSeconds())).addKeyValue("retryAfterSeconds", REFRESH_DELAY_STRING).addKeyValue("expired", tte.isNegative());
    }
}

