package com.aliyun.openservices.iot.api.message.impl;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import com.aliyun.openservices.iot.api.Profile;
import com.aliyun.openservices.iot.api.exception.IotClientException;
import com.aliyun.openservices.iot.api.http2.IotHttp2Client;
import com.aliyun.openservices.iot.api.http2.callback.AbstractHttp2StreamDataReceiver;
import com.aliyun.openservices.iot.api.http2.connection.Connection;
import com.aliyun.openservices.iot.api.http2.connection.ConnectionListener;
import com.aliyun.openservices.iot.api.http2.connection.ConnectionStatus;
import com.aliyun.openservices.iot.api.http2.entity.Http2Response;
import com.aliyun.openservices.iot.api.http2.entity.StreamData;
import com.aliyun.openservices.iot.api.message.api.MessageClient;
import com.aliyun.openservices.iot.api.message.callback.ConnectionCallback;
import com.aliyun.openservices.iot.api.message.callback.MessageCallback;
import com.aliyun.openservices.iot.api.message.callback.MessageCallback.Action;
import com.aliyun.openservices.iot.api.message.entity.Message;
import com.aliyun.openservices.iot.api.message.entity.MessageToken;
import com.aliyun.openservices.iot.api.message.entity.SubscribeInfo;
import com.aliyun.openservices.iot.api.util.StringUtil;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http2.DefaultHttp2Headers;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2Settings;
import io.netty.handler.codec.http2.Http2Stream;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * message client implement
 *
 * @author brhao
 * @date 16/03/2018
 */
@Slf4j
public class MessageClientImpl extends AbstractHttp2StreamDataReceiver implements ConnectionListener,
    MessageClient {
    private static final String MESSAGE_ID = "x-message-id";
    private static final String PATH_MESSAGE_ACK = "/message/ack";
    private static final String PATH_MESSAGE_PUB = "/message/pub";
    private static final String PATH_MESSAGE_SUB = "/message/sub";
    private static final String PATH_MESSAGE_UNSUB = "/message/unsub";
    private static final String PATH_CONNECT = "/message/echo/success";
    private static final String SLASH = "/";
    private static final String PLUS_SIGN = "+";
    private static final String CROSSHATCH = "#";
    private static final String IOT_ID = "x-iot-id";
    private static final String TOPIC = "x-topic";
    private static final String QOS = "x-qos";
    private static final String GENERATE_TIME = "x-generate-time";
    private static final String AS_STATUS_PREFIX = "/as/mqtt/status/";
    private final Profile profile;

    @Getter
    @Setter
    private MessageCallback messageCallback;
    private ExecutorService messageCallbackExecutorService;
    private ScheduledExecutorService publishRetryExecutorService;
    private IotHttp2Client client;
    private List<SubscribeInfo> subscriptionInfo;
    private AtomicBoolean started;
    private AtomicBoolean isConnected;
    private AtomicLong localMessageId;
    private ConnectionCallback connectionCallback;

    public MessageClientImpl(Profile profile) {
        this.profile = profile;
        this.subscriptionInfo = new CopyOnWriteArrayList<>();
        this.started = new AtomicBoolean(false);
        this.isConnected = new AtomicBoolean(false);
        this.localMessageId = new AtomicLong(0);
    }

    private CompletableFuture<Http2Response> sendAuth(Connection connection) {
        Http2Headers headers = client.authHeader();
        headers.path(PATH_CONNECT);
        headers.add("x-clear-session", profile.isCleanSession() ? "1" : "0");
        return client.sendRequest(connection, headers, null);
    }

    private void doConnect() {
        try {
            Connection connection = client.newConnection();

            connection.setDefaultStreamListener(this);

            CompletableFuture<Http2Response> completableFuture = sendAuth(connection);
            Http2Response r = completableFuture.get(15, TimeUnit.SECONDS);
            if (r.getStatus() != HttpResponseStatus.OK) {
                throw new IotClientException("connect to server failed", r);
            }
            connection.setStatus(ConnectionStatus.AUTHORIZED);
            client.addConnectionListener(this);
            isConnected.set(true);
            Optional.ofNullable(connectionCallback)
                .ifPresent(callback -> messageCallbackExecutorService.execute(() -> {
                    callback.onConnected(false);
                }));

        } catch (ExecutionException e) {
            throw new IotClientException(e.getCause());
        } catch (InterruptedException | TimeoutException e) {
            throw new IotClientException(e);
        }
    }

    private Message convertStreamData2Message(Http2Response response) throws IOException {
        if (!HttpResponseStatus.OK.equals(response.getStatus())) {
            throw new IOException("status is not success, code: " + response.getStatus() + ", content: " +
                new String(response.getContent()));
        }

        byte[] payload = response.getContent();
        String messageId = response.getHeaders().get(MESSAGE_ID).toString();
        String topic = null;
        if (response.getHeaders().contains(TOPIC)) {
            topic = response.getHeaders().get(TOPIC).toString();
        }
        int qos = 0;
        if (response.getHeaders().contains(QOS)) {
            qos = response.getHeaders().getInt(QOS);
        }

        long generateTime = 0;
        if (response.getHeaders().contains(GENERATE_TIME)) {
            generateTime = response.getHeaders().getLong(GENERATE_TIME);
        }

        // 消息回调
        if (isStatusCallback(topic)) {
            String[] splits  = topic.split("/");
            JSONObject jsonPayload =  JSON.parseObject(new String(payload));
            jsonPayload.remove("meta");
            if (splits.length > 5) {
                jsonPayload.put("productKey", splits[4]);
                jsonPayload.put("deviceName", splits[5]);
            }
            payload = jsonPayload.toJSONString().getBytes();
        }

        return new Message(payload, topic, messageId, qos, generateTime);
    }

    private boolean isStatusCallback(String topic) {
        return StringUtil.isNotEmpty(topic) && topic.startsWith(AS_STATUS_PREFIX);
    }

    private boolean needAck(Message m) {
        return (m.getQos() == 1 || m.getQos() == 2);
    }

    @Override
    public void onStreamError(Connection connection, Http2Stream stream, IOException e) {
        log.error("message receive error, {}", e.getMessage());
    }

    @Override
    public void onDataRead(Connection connection, Http2Stream stream, StreamData streamData) {
        try {
            Http2Response response = new Http2Response(streamData.getHeaders(), streamData.readAllData());
            Message m = convertStreamData2Message(response);
            MessageToken token = new MessageToken(m, connection, client);
            log.info("receive msg, messageId:{}, data size: {}", m.getMessageId(), m.getPayload().length);

            CompletableFuture<Action> cf = CompletableFuture.supplyAsync(() -> {
                try {
                    Optional<MessageCallback> op = subscriptionInfo.stream()
                        .filter(info -> isMatch(m.getTopic(), info.getTopic()))
                        .findFirst().map(SubscribeInfo::getCallback);

                    MessageCallback callback = op.orElse(messageCallback);
                    if (callback != null) {
                        return callback.consume(token);
                    }
                    log.warn("no message callback for " + m.getTopic());
                    return Action.CommitFailure;
                } catch (Throwable t) {
                    log.error("message consume error, messageId:{}, {}", m.getMessageId(), t);
                    return Action.CommitFailure;
                }
            }, messageCallbackExecutorService);

            if (needAck(m)) {
                cf.whenComplete((action, throwable) -> {
                    if (throwable != null) {
                        log.error("consume message {}, occurs error: ", m, throwable.getMessage());
                    }

                    log.info("consume message: {} , result: {}", m, action == null ? "null" : action.name());

                    if (action == Action.CommitSuccess) {
                        ack(token);
                    }
                });
            }
        } catch (IOException e) {
            log.error("message receive error,{}, {}", e.getMessage(), streamData);
        }
    }

    @Override
    public void onSettingReceive(Connection connection, Http2Settings settings) {
        // ignore
    }

    @Override
    public void onStatusChange(ConnectionStatus status, Connection connection) {
        if (status == ConnectionStatus.CREATED) {
            connection.setDefaultStreamListener(this);
            sendAuth(connection).whenComplete((http2Response, t) -> {
                if (t != null) {
                    log.error("failed to auth connection {}, {}", connection, t.getMessage());
                    connection.close();
                    return;
                }

                if (http2Response.getStatus() != HttpResponseStatus.OK) {
                    log.error("failed to auth connection {}, code: {}, content: {}, request id: {}", connection,
                        http2Response.getStatus().code(), new String(http2Response.getContent()),
                        http2Response.getRequestId());
                    connection.close();
                    return;
                }

                connection.setStatus(ConnectionStatus.AUTHORIZED);
            });
        }

        boolean hasAuthorizedConnection = client.allConnections().stream().anyMatch(Connection::isAuthorized);
        boolean isConnectedBefore = isConnected.get();
        if (!hasAuthorizedConnection) {
            isConnected.set(false);
        }

        Optional.ofNullable(connectionCallback)
            .ifPresent(callback -> {
                if (status == ConnectionStatus.AUTHORIZED) {
                    callback.onConnected(true);
                    return;
                }

                if (!hasAuthorizedConnection && isConnectedBefore) {
                    callback.onConnectionLost();
                }
            });
    }

    private boolean isMatch(String topicFullName, String topicFilter) {
        String[] nameItems = topicFullName.split(SLASH);
        String[] filterItems = topicFilter.split(SLASH);

        if (nameItems.length < filterItems.length) {
            return false;
        }

        if (nameItems.length != filterItems.length && !CROSSHATCH.equals(filterItems[filterItems.length - 1])) {
            return false;
        }

        for (int i = 0; i < filterItems.length; i++) {
            if (!CROSSHATCH.equals(filterItems[i]) && !PLUS_SIGN.equals(filterItems[i]) && !nameItems[i].equals(
                filterItems[i])) {
                return false;
            }
        }
        return true;
    }

    @Override
    public void connect(MessageCallback messageCallback) {
        if (started.compareAndSet(false, true)) {
            setMessageCallback(messageCallback);
            messageCallbackExecutorService = new ThreadPoolExecutor(profile.getCallbackThreadCorePoolSize(),
                profile.getCallbackThreadMaximumPoolSize(), 60, TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(profile.getCallbackThreadBlockingQueueSize()),
                new ThreadFactoryBuilder().setDaemon(true).setNameFormat("iot-message-client-receiver-%d").build());
            publishRetryExecutorService = new ScheduledThreadPoolExecutor(1, new ThreadFactoryBuilder().setNameFormat(
                "iot-message-client-schedule-thread").build());
            this.client = new IotHttp2Client(profile, profile.isMultiConnection() ? -1 : 1);
            doConnect();
        } else {
            throw new IotClientException("client is already connected!");
        }
    }

    @Override
    public void disconnect() {
        checkStarted();
        if (started.compareAndSet(true, false)) {
            client.removeConnectionListener(this);
            client.shutdown();
            messageCallbackExecutorService.shutdown();
            publishRetryExecutorService.shutdown();
            client = null;
            messageCallbackExecutorService = null;
            subscriptionInfo.clear();
            started.set(false);
        }
    }

    @Override
    public void setMessageListener(MessageCallback messageCallback) {
        this.messageCallback = messageCallback;
    }

    @Override
    public void setMessageListener(String topic, MessageCallback messageCallback) {
        if (StringUtil.isEmpty(topic)) {
            throw new IllegalArgumentException("topic can't be null");
        }

        Optional<SubscribeInfo> optional = subscriptionInfo.stream()
            .filter(info -> topic.equals(info.getTopic())).findAny();
        if (optional.isPresent()) {
            if (messageCallback == null) {
                subscriptionInfo.remove(optional.get());
            } else {
                optional.get().setCallback(messageCallback);
            }
        } else {
            subscriptionInfo.add(new SubscribeInfo(topic, messageCallback));
        }
    }

    @Override
    public CompletableFuture<Boolean> subscribe(String topic) {
        checkStarted();
        Http2Headers headers = new DefaultHttp2Headers();
        headers.path(PATH_MESSAGE_SUB + topic);
        return sendWithResult(headers, null, "failed to subscribe " + topic);
    }

    private CompletableFuture<Boolean> sendWithResult(Http2Headers headers, byte[] data,
                                                      String errorMessage) {
        Connection connection = getConnection();
        CompletableFuture<Boolean> future = new CompletableFuture<>();
        try {
            client.sendRequest(connection, headers, data).whenComplete(
                (response, throwable) -> {
                    if (throwable != null) {
                        log.error(errorMessage, throwable);
                        future.completeExceptionally(throwable);
                        return;
                    }
                    if (!HttpResponseStatus.OK.equals(response.getStatus())) {
                        IotClientException exception = new IotClientException(errorMessage, response);
                        log.error(exception.getMessage());
                        future.completeExceptionally(exception);
                    } else {
                        future.complete(true);
                    }
                });
        } catch (Exception e) {
            future.completeExceptionally(new IotClientException(errorMessage, e));
        }
        return future;
    }

    @Override
    public CompletableFuture<Boolean> subscribe(String topic, MessageCallback messageCallback) {
        checkStarted();
        setMessageListener(topic, messageCallback);
        return subscribe(topic);
    }

    @Override
    public CompletableFuture<Boolean> unsubscribe(String topic) {
        checkStarted();
        Http2Headers headers = new DefaultHttp2Headers();
        headers.path(PATH_MESSAGE_UNSUB + topic);
        return sendWithResult(headers, null, "failed to unsubscribe " + topic);
    }

    @Override
    public MessageToken publish(String topic, Message message) {
        checkStarted();
        Http2Headers headers = new DefaultHttp2Headers();
        headers.path(PATH_MESSAGE_PUB + topic);
        headers.add(QOS, String.valueOf(message.getQos()));
        MessageToken messageToken = new MessageToken(message, null, client);
        messageToken.setLocalMessageId(localMessageId.getAndIncrement());
        doPublish(messageToken, headers);
        return messageToken;
    }

    @Override
    public CompletableFuture<Boolean> ack(MessageToken messageToken) {
        Http2Headers headers = new DefaultHttp2Headers();
        String messageId = messageToken.getMessage().getMessageId();
        headers.set("x-message-id", messageId);
        headers.path(PATH_MESSAGE_ACK);
        client.sendRequest(messageToken.getConnection(), headers, null).whenComplete((r, t) -> {
            if (t != null) {
                log.error("ack failed, messageId {}, error {}", messageId, t.getMessage());
                return;
            }
            if (HttpResponseStatus.OK.equals(r.getStatus())) {
                log.debug("ack success, messageId {}", messageId);
            } else {
                log.error(new IotClientException("ack message: " + messageId + " failed", r).getMessage());
            }
        });
        return null;
    }

    @Override
    public boolean isConnected() {
        return isConnected.get();
    }

    @Override
    public void setConnectionCallback(ConnectionCallback connectionCallback) {
        this.connectionCallback = connectionCallback;
    }

    private void doPublish(MessageToken messageToken, Http2Headers headers) {
        log.info("publish message {}", messageToken.getLocalMessageId());
        try {
            client.sendRequest(getConnection(), headers, messageToken.getMessage().getPayload())
                .whenComplete((response, throwable) -> {
                    Exception exception = null;
                    if (throwable != null) {
                        exception = new IotClientException("failed to publish, message id: {}", throwable);
                    } else if (response == null) {
                        exception = new IotClientException("failed to publish, response is null");
                    } else if (!HttpResponseStatus.OK.equals(response.getStatus())) {
                        exception = new IotClientException("failed to publish", response);
                    }
                    if (exception == null) {
                        try {
                            Message m = convertStreamData2Message(response);
                            messageToken.getPublishFuture().complete(m);
                            return;
                        } catch (Exception e) {
                            log.error("failed to receive response, error: {}", e.getMessage(), e);
                        }
                    }
                    retryPublish(messageToken, headers, exception, false);
                });
        } catch (Exception e) {
            retryPublish(messageToken, headers,
                new IotClientException("failed to publish message, error msg:" + e.getMessage()), true);
        }
    }

    private void retryPublish(MessageToken messageToken, Http2Headers headers, Exception exception,
                              boolean isLocalError) {
        log.error("failed to publish message {}, error: {}", messageToken.getLocalMessageId(), exception.getMessage());
        if (messageToken.shouldStop(isLocalError)) {
            log.info("give up publishing, message id: {}", messageToken.getLocalMessageId());
            messageToken.getPublishFuture().completeExceptionally(exception);
        } else {
            messageToken.increaseAttemptCount();
            publishRetryExecutorService.schedule(() -> doPublish(messageToken, headers),
                messageToken.computeSleepTime(), TimeUnit.MILLISECONDS);
            log.info("message {} will be delivery after {} ms", messageToken.getLocalMessageId(),
                messageToken.computeSleepTime());
        }
    }

    private Connection getConnection() {
        return client.randomConnection(Connection::isAuthorized).orElseThrow(() ->
            new IotClientException("fail to publish, no connection exists"));
    }

    private void checkStarted() {
        if (!started.get()) {
            throw new IotClientException("client is not connected, please connect first");
        }
    }
}
