package com.aliyun.openservices.iot.api.message.entity;

import com.aliyun.openservices.iot.api.http2.IotHttp2Client;
import com.aliyun.openservices.iot.api.http2.connection.Connection;
import com.github.rholder.retry.*;
import lombok.Getter;
import lombok.Setter;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author brhao
 * @date 13/06/2018
 */
public class MessageToken {
    @Getter
    Message message;

    @Getter
    private Connection connection;
    @Getter
    private IotHttp2Client client;

    /**
     * stop strategy for qos1 message
     */
    @Getter
    @Setter
    private StopStrategy stopStrategy;
    /**
     * wait strategy for qos1 message
     */
    @Getter
    @Setter
    private WaitStrategy waitStrategy;

    private MessageAttempt attempt;

    @Getter
    @Setter
    private long localMessageId;

    @Getter
    private CompletableFuture<Message> publishFuture;

    public MessageToken(Message message, Connection connection, IotHttp2Client client) {
        this.message = message;
        this.connection = connection;
        this.client = client;
        this.stopStrategy = StopStrategies.stopAfterAttempt(32);
        this.waitStrategy = WaitStrategies.exponentialWait(10, TimeUnit.MINUTES);
        this.attempt = new MessageAttempt();
        this.publishFuture = new CompletableFuture<>();
    }

    /**
     * check if should retry when publish failed
     *
     * @param isLocalError if publish failed because of internal error.
     *                     It means message hasn't been sent and always should retry.
     * @return should retry
     */
    public boolean shouldStop(boolean isLocalError) {
        if (message.getQos() < 1 && !isLocalError) {
            return true;
        }
        return stopStrategy.shouldStop(attempt);
    }

    public long computeSleepTime() {
        return waitStrategy.computeSleepTime(attempt);
    }

    public void increaseAttemptCount() {
        attempt.attemptCount.incrementAndGet();
    }

    private static final class MessageAttempt implements Attempt {
        private long firstAttemptTime;
        private AtomicInteger attemptCount = new AtomicInteger(0);

        public MessageAttempt() {
            this.firstAttemptTime = System.currentTimeMillis();
        }

        @Override
        public Object get() throws ExecutionException {
            return null;
        }

        @Override
        public boolean hasResult() {
            return false;
        }

        @Override
        public boolean hasException() {
            return false;
        }

        @Override
        public Object getResult() throws IllegalStateException {
            return null;
        }

        @Override
        public Throwable getExceptionCause() throws IllegalStateException {
            return null;
        }

        @Override
        public long getAttemptNumber() {
            return attemptCount.get();
        }

        @Override
        public long getDelaySinceFirstAttempt() {
            return System.currentTimeMillis() - firstAttemptTime;
        }
    }
}
