/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.cloud.context.ans;

import com.alibaba.ans.core.NamingService;
import com.alibaba.ans.shaded.com.taobao.vipserver.client.utils.UtilAndComs;
import com.alibaba.cloud.context.AliCloudConfiguration;
import com.alibaba.cloud.context.AliCloudServerMode;
import com.alibaba.cloud.context.edas.*;
import com.alibaba.cloud.context.utils.AssertUtils;
import com.aliyuncs.edas.model.v20170801.GetSecureTokenResponse;
import com.aliyuncs.edas.model.v20170801.InsertApplicationResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * ANS初始化工具类
 *
 * @author xiaolongzuo
 */
public class AliCloudAnsInitializer {

    private static final Logger log = LoggerFactory
        .getLogger(AliCloudAnsInitializer.class);

    private AliCloudAnsInitializer() {
    }

    /**
     * 初始化ANS配置，执行完该方法以后，可以直接使用{@link NamingService}的方法。
     *
     * @param aliCloudConfiguration 阿里云的通用配置
     * @param edasConfiguration     EDAS的通用配置
     * @param ansConfiguration      ANS配置
     * @see NamingService
     */
    public static void initialize(AliCloudConfiguration aliCloudConfiguration,
                                  EdasConfiguration edasConfiguration, AnsConfiguration ansConfiguration) {
        initialize(aliCloudConfiguration, edasConfiguration, ansConfiguration, null);
    }

    public static void initialize(AliCloudConfiguration aliCloudConfiguration,
                                  EdasConfiguration edasConfiguration, AnsConfiguration ansConfiguration,
                                  AliCloudEdasSdk aliCloudEdasSdk) {
        EdasChangeOrderConfiguration edasChangeOrderConfiguration = EdasChangeOrderConfigurationFactory
            .getEdasChangeOrderConfiguration();
        if (edasChangeOrderConfiguration.isEdasManaged()) {
            log.info("Initialize ans from edas change order.");
            initializeFromEdasChangeOrder(edasChangeOrderConfiguration);
        } else if (ansConfiguration.getServerMode() == AliCloudServerMode.EDAS || edasConfiguration.isEnabled()) {
            log.info("Initialize ans from pop api.");
            initializeFromPopApi(aliCloudConfiguration, edasConfiguration,
                aliCloudEdasSdk);
        } else {
            log.info("Initialize ans from ans configuration.");
            initializeFromAnsConfiguration(ansConfiguration);
        }
    }

    private static void initializeFromEdasChangeOrder(EdasChangeOrderConfiguration edasChangeOrderConfiguration) {
        System.setProperty(UtilAndComs.PROPERTY_NAME_APP, edasChangeOrderConfiguration.getProjectName());
        System.setProperty(UtilAndComs.PROPERTY_NAME_TENANT_ID, edasChangeOrderConfiguration.getTenantId());
        System.setProperty(UtilAndComs.PROPERTY_NAME_AK, edasChangeOrderConfiguration.getDauthAccessKey());
        System.setProperty(UtilAndComs.PROPERTY_NAME_SK, edasChangeOrderConfiguration.getDauthSecretKey());
        System.setProperty("address.server.port", edasChangeOrderConfiguration.getAddressServerPort());
        System.setProperty("address.server.domain", edasChangeOrderConfiguration.getAddressServerDomain());
    }

    private static void initializeFromPopApi(AliCloudConfiguration aliCloudConfiguration,
                                             EdasConfiguration edasConfiguration, AliCloudEdasSdk aliCloudEdasSdk) {
        AssertUtils.notNull(aliCloudConfiguration.getAccessKey(), "Access key can't be null.");
        AssertUtils.notNull(aliCloudConfiguration.getSecretKey(), "Secret key can't be null.");
        AssertUtils.notNull(edasConfiguration.getNamespace(), "Namespace can't be null.");
        AssertUtils.isTrue(edasConfiguration.isApplicationNameValid(), "Application name can't be empty.");
        if (aliCloudEdasSdk == null) {
            aliCloudEdasSdk = AliCloudEdasSdkFactory.getDefaultAliCloudEdasSdk(aliCloudConfiguration,
                edasConfiguration.getRegionId());
        }
        GetSecureTokenResponse.SecureToken secureToken = aliCloudEdasSdk.getSecureToken(
            edasConfiguration.getNamespace());
        InsertApplicationResponse.ApplicationInfo applicationInfo = aliCloudEdasSdk
            .getApplicationInfo(edasConfiguration.getApplicationName(),
                edasConfiguration.getNamespace());

        System.setProperty(UtilAndComs.PROPERTY_NAME_TENANT_ID, secureToken.getTenantId());
        System.setProperty(UtilAndComs.PROPERTY_NAME_AK, secureToken.getAccessKey());
        System.setProperty(UtilAndComs.PROPERTY_NAME_SK, secureToken.getSecretKey());
        System.setProperty("address.server.domain", secureToken.getAddressServerHost());
        System.setProperty(UtilAndComs.PROPERTY_NAME_APP, applicationInfo.getAppId());
    }

    private static void initializeFromAnsConfiguration(
        AnsConfiguration ansConfiguration) {
        System.setProperty("com.alibaba.ans.shaded.com.taobao.vipserver.serverlist", ansConfiguration.getServerList());
        System.setProperty("vipserver.server.port", ansConfiguration.getServerPort());
    }

}
