/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.cloud.context.acm;

import com.alibaba.cloud.context.AliCloudConfiguration;
import com.alibaba.cloud.context.AliCloudServerMode;
import com.alibaba.cloud.context.edas.EdasChangeOrderConfiguration;
import com.alibaba.cloud.context.edas.EdasChangeOrderConfigurationFactory;
import com.alibaba.cloud.context.edas.EdasConfiguration;
import com.alibaba.cloud.context.utils.AssertUtils;
import com.alibaba.edas.acm.ConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Properties;

/**
 * ACM初始化工具类
 *
 * @author xiaolongzuo
 */
public class AliCloudAcmInitializer {

    private static final Logger log = LoggerFactory
        .getLogger(AliCloudAcmInitializer.class);

    private AliCloudAcmInitializer() {
    }

    /**
     * 初始化ACM配置，执行完该方法以后，可以直接使用{@link ConfigService}的方法。
     *
     * @param aliCloudConfiguration 阿里云的通用配置
     * @param edasConfiguration     EDAS的通用配置
     * @param acmConfiguration      ACM配置
     * @see ConfigService
     */
    public static void initialize(AliCloudConfiguration aliCloudConfiguration,
                                  EdasConfiguration edasConfiguration, AcmConfiguration acmConfiguration) {
        Properties properties = new Properties();
        EdasChangeOrderConfiguration edasChangeOrderConfiguration = EdasChangeOrderConfigurationFactory
            .getEdasChangeOrderConfiguration();
        if (edasChangeOrderConfiguration.isEdasManaged()) {
            log.info("Initialize acm from edas change order.");
            initializeFromEdasChangeOrder(properties, edasChangeOrderConfiguration);
        } else if (edasConfiguration.isEnabled() || acmConfiguration.getServerMode() == AliCloudServerMode.EDAS) {
            log.info("Initialize acm from pop api.");
            initializeFromPopApi(properties, aliCloudConfiguration, acmConfiguration);
        } else {
            log.info("Initialize acm from acm configuration.");
            initializeFromAcmConfiguration(acmConfiguration);
        }
        ConfigService.init(properties);
    }

    private static void initializeFromEdasChangeOrder(Properties properties,
                                                      EdasChangeOrderConfiguration edasChangeOrderConfiguration) {
        properties.put("endpoint", edasChangeOrderConfiguration.getAddressServerDomain());
        properties.put("namespace", edasChangeOrderConfiguration.getTenantId());
        properties.put("accessKey", edasChangeOrderConfiguration.getDauthAccessKey());
        properties.put("secretKey", edasChangeOrderConfiguration.getDauthSecretKey());
    }

    private static void initializeFromPopApi(Properties properties,
                                             AliCloudConfiguration aliCloudConfiguration,
                                             AcmConfiguration acmConfiguration) {
        AssertUtils.notNull(aliCloudConfiguration.getAccessKey(), "Access key can't be null.");
        AssertUtils.notNull(aliCloudConfiguration.getSecretKey(), "Secret key can't be null.");
        AssertUtils.notNull(acmConfiguration.getNamespace(), "Namespace can't be null.");
        AssertUtils.notNull(acmConfiguration.getEndpoint(), "Endpoint can't be empty.");

        properties.put("endpoint", acmConfiguration.getEndpoint());
        properties.put("namespace", acmConfiguration.getNamespace());
        properties.put("accessKey", aliCloudConfiguration.getAccessKey());
        properties.put("secretKey", aliCloudConfiguration.getSecretKey());
    }

    private static void initializeFromAcmConfiguration(
        AcmConfiguration acmConfiguration) {
        if (null != acmConfiguration.getServerList()) {
            System.setProperty("DIAMOND.SERVER.IPS", acmConfiguration.getServerList());
        }
        if (null != acmConfiguration.getServerPort()) {
            System.setProperty("diamond.server.port", acmConfiguration.getServerPort());
        }
    }

}
