package cn.smarthse.rho.core.framework.utils;

/**
 * 判断一个点 (x, y) 是否位于由四个点 (x1,y1), (x2,y2), (x3,y3), (x4,y4) 构成的多边形内部
 *
 * 这个实现中，我们使用了叉积（Cross Product）来更精确地判断点相对于边的位置。叉积的正负性可以告诉我们一个点是在边的哪一侧。
 * 如果叉积为0，说明点在边上或是一个顶点。
 * 另外，我们还检查了点是否在多边形的边上或顶点上，如果是，我们直接返回true，因为题目要求判断点是否在多边形区域内，包括边界。
 * 请注意，如果多边形不是简单的（即它有自交的边），那么这个方法可能无法正确工作。
 * 在这种情况下，您可能需要一个更复杂的算法来分解多边形为简单的子多边形，并分别检查每个子多边形。
 * @author Wuxq
 * @since  2024/6/27
 */
public class PolygonAreaUtil {


    // 使用叉积（Cross Product）来判断点的位置
    private static double crossProduct(double x1, double y1, double x2, double y2, double x3, double y3) {
        return (x2 - x1) * (y3 - y1) - (y2 - y1) * (x3 - x1);
    }

    public static boolean isPointInsidePolygon(double x, double y, double[] xPoints, double[] yPoints) {
        int numVertices = xPoints.length;
        if (numVertices != yPoints.length || numVertices < 3) {
            throw new IllegalArgumentException("The number of points must be at least 3 and equal in both arrays.");
        }

        boolean inside = false;
        for (int i = 0, j = numVertices - 1; i < numVertices; j = i++) {
            // 如果点在多边形的边上或顶点上，直接返回true
            if ((yPoints[i] == y && (xPoints[j] > x || (xPoints[j] == x && xPoints[i] <= x))) ||
                    (yPoints[j] == y && (xPoints[i] > x || (xPoints[i] == x && xPoints[j] <= x))) ||
                    (y > Math.min(yPoints[i], yPoints[j]) && y < Math.max(yPoints[i], yPoints[j]) &&
                            x < Math.max(xPoints[i], xPoints[j]) && crossProduct(xPoints[i], yPoints[i], xPoints[j], yPoints[j], x, y) == 0)) {
                return true;
            }

            if (yPoints[i] > y != yPoints[j] > y && x < (xPoints[j] - xPoints[i]) * (y - yPoints[i]) / (yPoints[j] - yPoints[i]) + xPoints[i]) {
                inside = !inside;
            }
        }

        return inside;
    }

    public static void main(String[] args) {
        double x1 = 0, y1 = 0;
        double x2 = 1, y2 = 0;
        double x3 = 1, y3 = 1;
        double x4 = 0, y4 = 1;

        double[] xPoints = {x1, x2, x3, x4};
        double[] yPoints = {y1, y2, y3, y4};

        double testX = 2.5, testY = 0.5;

        boolean inside = isPointInsidePolygon(testX, testY, xPoints, yPoints);
        System.out.println("Is point (" + testX + ", " + testY + ") inside the polygon? " + inside);
    }
}
