package cn.smarthse.rho.core.framework.utils;

import cn.smarthse.rho.core.framework.model.LayuiGridData;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;

import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.function.Supplier;

/**
 * easyexcel工具类
 *
 * @Author liaoly(廖凌云) [1302013247@qq.com]
 * @Date 2024/6/7 14:46
 */
@Slf4j
public class EasyExcelUtil {

    /**
     * 循环分页查询导出excel （后期可以考虑多线程进行，写入时用锁存器）
     *
     * @param includeColumnFiledNames 需要导出的列字段名 eg：{"name", "age"}
     * @param fileName                excel文件名 不带文件后缀
     * @param clazz                   数据类型
     * @param response
     * @param supplier                分页查询方法
     * @author liaoly(廖凌云) [1302013247@qq.com]
     * @date 2024/6/7 15:55
     */
    public static void exportDynamicExcel(List<String> includeColumnFiledNames, String fileName, Class<?> clazz, HttpServletResponse response, Supplier<LayuiGridData<?>> supplier) {
        ServletOutputStream outputStream;
        ExcelWriter excelWriter = null;
        try {
            outputStream = response.getOutputStream();
            response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
            response.setCharacterEncoding(StandardCharsets.UTF_8.name());
            String encodeFileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8).replaceAll("\\+", "%20");
            response.setHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment;filename=" + encodeFileName + ".xlsx");

            ExcelWriterBuilder excelBuilder = EasyExcel.write(outputStream, clazz);
            includeColumnFiledName(includeColumnFiledNames, excelBuilder);
            excelWriter = excelBuilder.build();

            export(excelWriter, supplier);
        } catch (IOException e) {
            log.error("分页导出Excel异常", e);
            throw new RuntimeException("Excel导出失败", e);
        } finally {
            if (excelWriter != null) {
                try {
                    excelWriter.close();
                } catch (Exception e) {
                    log.error("Excel writer close failed", e);
                }
            }
        }
    }

    /**
     * 使用EasyExcel导出动态列的Excel
     *
     * @param includeColumnFiledNames 需要导出的列字段名 eg：{"name", "age"}
     * @param fileName                excel文件名 不带文件后缀
     * @param list                    数据集合 eg: List<AStatisticsRiskVo> list
     * @param clazz                   数据类型
     * @param response
     */
    public static void exportDynamicExcel(List<String> includeColumnFiledNames, String fileName, List<?> list, Class<?> clazz, HttpServletResponse response) {
        try {
            ServletOutputStream os = response.getOutputStream();
            response.setContentType(MediaType.APPLICATION_OCTET_STREAM_VALUE);
            response.setCharacterEncoding(StandardCharsets.UTF_8.name());
            String encodeFileName = URLEncoder.encode(fileName, StandardCharsets.UTF_8).replaceAll("\\+", "%20");
            response.setHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment;filename=" + encodeFileName + ".xlsx");

            ExcelWriterBuilder excelBuilder = EasyExcel.write(os, clazz);
            excelBuilder = includeColumnFiledName(includeColumnFiledNames, excelBuilder);
            excelBuilder.useDefaultStyle(false).sheet("Sheet1").doWrite(list);
            os.close();
        } catch (IOException e) {
            log.error("导出Excel异常", e);
        }
    }

    /**
     * 输出列处理
     *
     * @param includeColumnFieldNames 包含的字段
     * @param excelBuilder
     * @return {@link ExcelWriterBuilder}
     * @author liaoly(廖凌云) [1302013247@qq.com]
     * @date 2024/6/7 15:00
     */
    private static ExcelWriterBuilder includeColumnFiledName(List<String> includeColumnFieldNames, ExcelWriterBuilder excelBuilder) {
        excelBuilder = excelBuilder.registerWriteHandler(new LongestMatchColumnWidthStyleStrategy());
        if (!includeColumnFieldNames.isEmpty()) {
            excelBuilder = excelBuilder.includeColumnFieldNames(includeColumnFieldNames);
        }
        return excelBuilder;
    }

    /**
     * 分页查询，并写入excel，直至分页完成
     *
     * @param excelWriter
     * @param supplier    分页查询接口
     * @author liaoly(廖凌云) [1302013247@qq.com]
     * @date 2024/6/7 17:47
     */
    private static void export(ExcelWriter excelWriter, Supplier<LayuiGridData<?>> supplier) {
        long page = 1; //当前页码
        long pages; //总页数
        try {
            do {
                LayuiGridData<?> res = supplier.get();
                excelWriter.write(res.getData(), EasyExcel.writerSheet("Sheet1").build());
                pages = res.getPages();
                page++;
            } while (page <= pages); // 修改为 <=
        } finally {
            if (excelWriter != null) {
                try {
                    excelWriter.finish();
                } catch (Exception e) {
                    log.error("Excel writer finish failed", e);
                }
            }
        }
    }

}
