package cn.smarthse.rho.core.framework.mybatis;

import cn.hutool.core.util.ObjectUtil;
import cn.smarthse.rho.core.framework.filter.dubbo.RhoUserInfoHolder;
import cn.smarthse.rho.core.framework.model.BaseEntity;
import com.baomidou.mybatisplus.core.handlers.MetaObjectHandler;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.reflection.MetaObject;

import javax.smarthse.framework.exception.ServiceException;
import java.time.LocalDateTime;
import java.util.Date;

/**
 * MP注入处理器
 */
@Slf4j
public class InjectionMetaObjectHandler implements MetaObjectHandler {

    /**
     * 插入填充方法，用于在插入数据时自动填充实体对象中的创建时间、更新时间、创建人、更新人等信息
     *
     * @param metaObject 元对象，用于获取原始对象并进行填充
     */
    @Override
    public void insertFill(MetaObject metaObject) {
        try {
            if (ObjectUtil.isNotNull(metaObject) && metaObject.getOriginalObject() instanceof BaseEntity) {
                BaseEntity baseEntity = (BaseEntity) metaObject.getOriginalObject();
                // 获取当前时间作为创建时间和更新时间，如果创建时间不为空，则使用创建时间，否则使用当前时间
                LocalDateTime current = ObjectUtil.isNotNull(baseEntity.getCreateDate()) ? baseEntity.getCreateDate() : LocalDateTime.now();
                baseEntity.setCreateDate(current);
                baseEntity.setUpdateDate(current);

                // 如果创建人为空，则填充当前登录用户的信息
                if (ObjectUtil.isNull(baseEntity.getCreateBy())) {
                    if (ObjectUtil.isNotNull(getLoginUserId())) {
                        // 填充创建人、更新人和创建部门信息
                        baseEntity.setCreateBy(getLoginUserId());
                        baseEntity.setUpdateBy(getLoginUserId());
                    }
                }
            } else {
                Date date = new Date();
                this.strictInsertFill(metaObject, "createTime", Date.class, date);
                this.strictInsertFill(metaObject, "updateTime", Date.class, date);
            }
        } catch (Exception e) {
            throw new ServiceException("自动注入异常 => " + e.getMessage());
        }
    }

    /**
     * 更新填充方法，用于在更新数据时自动填充实体对象中的更新时间和更新人信息
     *
     * @param metaObject 元对象，用于获取原始对象并进行填充
     */
    @Override
    public void updateFill(MetaObject metaObject) {
        try {
            if (ObjectUtil.isNotNull(metaObject) && metaObject.getOriginalObject() instanceof BaseEntity) {
                BaseEntity baseEntity = (BaseEntity) metaObject.getOriginalObject();
                // 获取当前时间作为更新时间，无论原始对象中的更新时间是否为空都填充
                baseEntity.setUpdateDate(LocalDateTime.now());
                // 获取当前登录用户的ID，并填充更新人信息
                if (ObjectUtil.isNotNull(getLoginUserId())) {
                    baseEntity.setUpdateBy(getLoginUserId());
                }
            }
        } catch (Exception e) {
            throw new ServiceException("自动注入异常 => " + e.getMessage());
        }
    }

    private Long getLoginUserId() {
        Long loginUserId;
        try {
            RhoUserInfoHolder holder = RhoUserInfoHolder.getCurrentUser();
            loginUserId = holder.getUserId();
        } catch (Exception e) {
            log.warn("自动注入警告 => 用户未登录");
            return null;
        }
        return loginUserId;
    }

}
