package cn.smarthse.rho.core.framework.mybatis;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.smarthse.framework.core.utils.SpringUtils;
import cn.smarthse.framework.mybatisPlus.config.properties.DataChangeRecorderProperties;
import cn.smarthse.framework.mybatisPlus.plugins.inner.DataChangeRecorderInnerInterceptor;
import cn.smarthse.framework.mybatisPlus.service.DataChangeRecorderService;
import cn.smarthse.rho.core.framework.Constant;
import cn.smarthse.rho.core.framework.filter.dubbo.RhoUserInfoHolder;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONException;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import lombok.extern.slf4j.Slf4j;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;

@Slf4j
public class DataChangeRecorderServiceImpl implements DataChangeRecorderService {

    private final String delColumn;
    private final Map<String, Map<String, String>> tableColumnMap;

    private DataChangeRecordMapper dataChangeRecordMapper;

    public DataChangeRecorderServiceImpl(DataChangeRecorderProperties dataChangeRecorderProperties) {
        this.delColumn = dataChangeRecorderProperties.getDelColumn();
        this.tableColumnMap = dataChangeRecorderProperties.getTableColumnMap();
    }

    @Override
    public List<?> convertToLogs(DataChangeRecorderInnerInterceptor.OperationResult operationResult) {

        String changedData = operationResult.getChangedData();
        if (StringUtils.isBlank(changedData)) {
            return Collections.emptyList();
        }

        JSONArray array;
        try {
            array = JSON.parseArray(changedData);
            //{"tableName":"sys_log","operation":"update","recordStatus":"true","cost(ms)":9,
            // "changedData":[{"ID":"1769880112666312705","USERNAME":"超神科技->超神科技2","TITLE":"登录->登录2","MODULE_TYPE":"0->2"}]}
        } catch (JSONException e) {
            log.error("JSON parse error, changedData={}", changedData, e);
            return Collections.emptyList();
        }

        if (CollUtil.isEmpty(array)) {
            return Collections.emptyList();
        }

        Map<String, String> columnMap = tableColumnMap.get(operationResult.getTableName());
        if (MapUtil.isEmpty(columnMap)) {
            return Collections.emptyList();
        }

        LocalDateTime now = LocalDateTime.now();
        RhoUserInfoHolder holder = RhoUserInfoHolder.getCurrentUser();

        String keyColumn = operationResult.getKeyColumn();
        String upperKeyColumn = keyColumn.toUpperCase();
        String operation = operationResult.getOperation();
        boolean isInsert = DataChangeRecorderProperties.OperationType.INSERT.name().equalsIgnoreCase(operation);

        List<SysChangeLog> logList = new ArrayList<>(array.size() * columnMap.size());

        for (Object o : array) {
            JSONObject jsonObject = (JSONObject) o;
            String id = jsonObject.getString(upperKeyColumn);
            if (StringUtils.isBlank(id)) {
                continue;
            }

            Long dataId;
            if (isInsert) {
                String[] contentSplit = id.split("->");
                dataId = Convert.toLong(contentSplit[1]);
                if (Objects.isNull(dataId)) {
                    continue;
                }
            } else {
                dataId = Convert.toLong(id);
            }

            long batchCode = Instant.now().toEpochMilli();

            for (String colomn : jsonObject.keySet()) {
                if (colomn.toUpperCase().equals(upperKeyColumn)) {
                    continue;
                }

                SysChangeLog log = new SysChangeLog();
                log.setBatchCode(batchCode);
                log.setTableName(operationResult.getTableName());
                log.setDataId(dataId);
                log.setFieldName(columnMap.get(colomn.toLowerCase()));
                log.setFieldValue(colomn.toLowerCase());

                String content = jsonObject.getString(colomn);
                String[] contentSplit = content.split("->");
                log.setBeforeValue(contentSplit.length >= 1 ? contentSplit[0] : null);
                log.setAfterValue(contentSplit.length >= 2 ? contentSplit[1] : null);

                log.setCid(holder.getOidOrCid());
                log.setUserId(holder.getUserId());
                log.setUsername(holder.getUserName());

                log.setOperation(colomn.equalsIgnoreCase(delColumn) ? "delete" : operationResult.getOperation());

                log.setEditTime(now);
                log.setContent(content);
                log.setCreateBy(log.getUserId());
                log.setCreateDate(now);
                log.setIsValid(Constant.ACTIVE_YES);

                logList.add(log);
            }
        }
        return logList;
    }

    @Override
    public void saveBatch(List<?> logs) {
        if (CollectionUtils.isEmpty(logs)) {
            return;
        }

        getDataChangeRecordMapper().insert((List<SysChangeLog>) logs);
    }

    private DataChangeRecordMapper getDataChangeRecordMapper() {
        if (ObjectUtil.isNull(dataChangeRecordMapper)) {
            synchronized (DataChangeRecorderInnerInterceptor.class) {
                if (ObjectUtil.isNull(dataChangeRecordMapper)) {
                    dataChangeRecordMapper = SpringUtils.getBean(DataChangeRecordMapper.class);
                }
            }
        }
        return dataChangeRecordMapper;
    }

}