package cn.smarthse.core.framework.utils.login;

import cn.hutool.core.util.StrUtil;
import cn.smarthse.core.framework.Constant;
import cn.smarthse.core.framework.utils.ServletUtils;
import cn.smarthse.core.framework.utils.StringUtils;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import org.slf4j.MDC;

import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.Objects;

/**
 * @Author liaoly(廖凌云) [1302013247@qq.com]
 * @Date 2022/2/20 14:16
 */
public class JwtUtil {

    public static Long getUserId() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("userId").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static Long getUserId(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("userId").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static String getUsername() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("username").asString();
        } catch (Exception e) {
            return null;
        }
    }

    public static String getUsername(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("username").asString();
        } catch (Exception e) {
            return null;
        }
    }

    public static String getFullName() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("fullName").asString();
        } catch (Exception e) {
            return null;
        }
    }

    public static String getFullName(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("fullName").asString();
        } catch (Exception e) {
            return null;
        }
    }

    public static Long getOid() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("oid").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static Long getOid(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("oid").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static Long getCid() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("cid").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static Long getCid(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("cid").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static Long getAreaId() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("areaId").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static Long getAreaId(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("areaId").asLong();
        } catch (Exception e) {
            return null;
        }
    }

    public static Integer getIsTest() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("isTest").asInt();
        } catch (Exception e) {
            return null;
        }
    }

    public static Integer getIsTest(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("isTest").asInt();
        } catch (Exception e) {
            return null;
        }
    }

    public static Integer getLoginType() {
        try {
            DecodedJWT jwt = JWT.decode(getToken());
            return jwt.getClaim("loginType").asInt();
        } catch (Exception e) {
            return null;
        }
    }

    public static Integer getLoginType(String token) {
        try {
            DecodedJWT jwt = JWT.decode(token);
            return jwt.getClaim("loginType").asInt();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取当前登录用户的token,如果 HttpServletRequest 为空的话，尝试从 MDC中获取；token为null则获取refreshToken
     *
     * @return token
     */
    public static String getToken() {

        HttpServletRequest request = ServletUtils.getRequest();
        String token;
        if (request == null) {
            token = MDC.get(Constant.TOKEN_HEADER_NAME);
        } else {
            token = request.getHeader(Constant.TOKEN_HEADER_NAME);
            if (StrUtil.isBlank(token)) {
                token = request.getParameter(Constant.REFRESH_TOKEN_HEADER_NAME);
            }
        }
        return token;
    }

    /**
     * 获取当前登录用户的token,如果token为null则获取refreshToken
     *
     * @param request HttpServletRequest
     * @return token
     */
    public static String getToken(HttpServletRequest request) {
        String token = request.getHeader(Constant.TOKEN_HEADER_NAME);
        if (StrUtil.isBlank(token)) {
            return request.getParameter(Constant.REFRESH_TOKEN_HEADER_NAME);
        } else {
            return token;
        }
    }

    /**
     * 创建token
     *
     * @param userId   用户id
     * @param username 用户名/手机号
     * @param secret   token验签密钥
     * @param time     token的有效时间 单位:毫秒
     * @return 加密的token
     */
    public static String createToken(Long userId, String username, String fullName, Long oid, Long cid, Long areaId, Integer isTest, Integer loginType, String secret, Long time) {
        Date date = new Date(System.currentTimeMillis() + time);
        Algorithm algorithm = Algorithm.HMAC256(secret);
        return JWT.create()
                .withClaim("userId", userId)
                .withClaim("username", username)
                .withClaim("fullName", fullName)
                .withClaim("oid", oid)
                .withClaim("cid", cid)
                .withClaim("areaId", areaId)
                .withClaim("isTest", isTest)
                .withClaim("loginType", loginType)
                .withExpiresAt(date).sign(algorithm);
    }

    /**
     * 根据token 刷新token（token快过期or已过期需用此方法得到新的token）
     *
     * @param token  token
     * @param secret token解密密钥
     * @return {@link String}
     * @author liaoly(廖凌云) [1302013247@qq.com]
     * @date 2024/5/27 16:54
     */
    public static String refreshToken(String token, String secret) {
        return createToken(getUserId(token), getUsername(token), getFullName(token), getOid(token), getCid(token), getAreaId(token), getIsTest(token), getLoginType(token), secret, Constant.TOKEN_EXPIRE_TIME);
    }

    /**
     * 校验token是否合法
     *
     * @param token  密钥
     * @param secret token解密密钥
     * @return 是否合法 合法返回true
     */
    public static boolean verify(String token, String secret) {

        if (StringUtils.isBlank(secret)) {
            return false;
        }
        try {
            // 根据密码生成JWT效验器
            Algorithm algorithm = Algorithm.HMAC256(secret);
            JWTVerifier verifier = JWT.require(algorithm)
                    .withClaim("username", getUsername(token))
                    .withClaim("userId", getUserId(token)).build();
            // 效验TOKEN
            verifier.verify(token);
            return true;
        } catch (JWTVerificationException exception) {
            return false;
        }
    }

    /**
     * 校验token是否过期
     *
     * @param token token
     * @return {@link Boolean} 过期返回true
     * @author liaoly
     * @date 2022/3/10 11:32
     */
    public static Boolean isTokenExpired(String token) {
        final Date expiration = getExpiration(token);
        return expiration == null || expiration.before(new Date());
    }

    /**
     * 获取token的过期时间
     *
     * @param token token
     * @return {@link Date}
     * @author liaoly
     * @date 2022/3/10 11:32
     */
    public static Date getExpiration(String token) {
        //token 就是jwt生成的token
        try {
            return JWT.decode(token).getClaims().get("exp").asDate();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取token的过期时间 单位为 秒
     *
     * @param token token
     * @return {@link Date}
     * @author liaoly
     * @date 2022/3/10 11:32
     */
    public static Long getExpirationLong(String token) {
        //token 就是jwt生成的token
        try {
            return JWT.decode(token).getClaims().get("exp").asLong();
        } catch (Exception e) {
            return 0L;
        }
    }

    /**
     * 获取token的过期时间 单位为 毫秒
     *
     * @param token token
     * @return {@link Date}
     * @author liaoly
     * @date 2022/3/10 11:32
     */
    public static Long getExpirationMillis(String token) {
        //token 就是jwt生成的token
        try {
            return JWT.decode(token).getClaims().get("exp").asLong() * 1000;
        } catch (Exception e) {
            return 0L;
        }
    }

    public static Boolean isOnlineSpecialSuperAndTestUser() {

        String loginUserName = getUsername();

        return "superuser".equalsIgnoreCase(loginUserName) || Objects.equals(getIsTest(), 1);
    }

}
