package cn.smarthse.core.framework.utils;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.extra.servlet.ServletUtil;
import cn.hutool.json.JSONUtil;
import org.springframework.http.HttpStatus;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * 客户端工具类
 *
 * @author lixiao
 */
public class ServletUtils {

	/**
	 * 获取request
	 */
	public static HttpServletRequest getRequest() {
		ServletRequestAttributes requestAttributes = getRequestAttributes();
		return requestAttributes == null ? null : requestAttributes.getRequest();
	}

	/**
	 * 获取response
	 */
	public static HttpServletResponse getResponse() {
		ServletRequestAttributes requestAttributes = getRequestAttributes();
		return requestAttributes == null ? null : requestAttributes.getResponse();
	}

	/**
	 * 获取session
	 */
	public static HttpSession getSession() {

		return getRequest().getSession();
	}

	public static ServletRequestAttributes getRequestAttributes() {
		RequestAttributes attributes = RequestContextHolder.getRequestAttributes();
		return (ServletRequestAttributes) attributes;
	}

	/**
	 * 将字符串渲染到客户端
	 * @param response 渲染对象
	 * @param string 待渲染的字符串
	 * @return null
	 */
	public static String renderString(HttpServletResponse response, String string) {
		response.setStatus(HttpStatus.OK.value());
		response.setCharacterEncoding(StandardCharsets.UTF_8.name());
		response.setContentType("application/json;charset=UTF-8");
		try (PrintWriter writer = response.getWriter()) {
			writer.print(string);
		} catch (IOException e) {
			// ignore
		}
		return null;
	}

	/**
	 * 获取当前项目的地址 eg：http://192.168.1.222:8010/admin （/admin 是contextPath）
	 * request.getScheme() + "://" + request.getServerName() + ":" + request.getServerPort() + request.getContextPath()
	 */
	public static String getCtx(HttpServletRequest request) {

		return request.getScheme() + "://" + request.getServerName() + ":" + request.getServerPort() + request.getContextPath();
	}

	/**
	 * 获取当前项目的地址 eg：http://192.168.1.222:8010/admin （/admin 是contextPath）
	 * request.getScheme() + "://" + request.getServerName() + ":" + request.getServerPort() + request.getContextPath()
	 */
	public static String getCtx() {
		HttpServletRequest request = getRequest();
		return request.getScheme() + "://" + request.getServerName() + ":" + request.getServerPort() + request.getContextPath();
	}

	/**
	 * 获取请求参数，返回json(先从request param中获取，如果没有再从rewuest body中获取. 注意：重写inputstream读取方法，避免不能重复读取)
	 */
	public static String getRequestParam(HttpServletRequest request) {
		if (request == null) return null;
		if (request instanceof MultipartFile) {
			return null;
		}
		Map<String, String[]> parameterMap = request.getParameterMap();

		return CollectionUtil.isNotEmpty(parameterMap) ? JSONUtil.toJsonStr(parameterMap) : ServletUtil.getBody(request);
	}

	/**
	 * 获取IP
	 * @param request
	 * @return
	 */
	public static String getIp(HttpServletRequest request) {
		String ip = request.getHeader("X-Forwarded-For");
		if (StringUtils.isNotEmpty(ip) && !"unKnown".equalsIgnoreCase(ip)) {
			//多次反向代理后会有多个ip值，第一个ip才是真实ip
			int index = ip.indexOf(",");
			if (index != -1) {
				return ip.substring(0, index);
			} else {
				return ip;
			}
		}
		ip = request.getHeader("X-Real-IP");
		if (StringUtils.isNotEmpty(ip) && !"unKnown".equalsIgnoreCase(ip)) {
			return ip;
		}
		return request.getRemoteAddr();
	}
}
