package cn.smarthse.core.framework.shiro;

import cn.smarthse.common.redis.utils.RedisUtils;
import cn.smarthse.core.framework.RedisKey;
import cn.smarthse.core.framework.bootstrap.properties.SysProperties;
import lombok.RequiredArgsConstructor;
import org.apache.shiro.authc.AuthenticationInfo;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.ExcessiveAttemptsException;
import org.apache.shiro.authc.LockedAccountException;
import org.apache.shiro.authc.credential.HashedCredentialsMatcher;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 密码匹配器
 *
 * @author liaoly(廖凌云) [1302013247@qq.com]
 * @date 2024/3/13 15:33
 */
@RequiredArgsConstructor
public class ShiroAuthorizingCredentialsMatcher extends HashedCredentialsMatcher {

    private final SysProperties sysProperties;

    @Override
    public boolean doCredentialsMatch(AuthenticationToken token, AuthenticationInfo info) {

        String keyName = RedisKey.passwordRetry((String) token.getPrincipal());

        AtomicInteger retryCacheCount = RedisUtils.getCacheObject(keyName);

        //需要锁定后输入正确密码还能登录就用">"，反之用">="
        if (retryCacheCount != null && retryCacheCount.get() >= retryCount()) {
            throw new LockedAccountException();
        }

        boolean matches = super.doCredentialsMatch(token, info);

        if (matches) {
            RedisUtils.deleteObject(keyName);
        } else {
            AtomicInteger updateRetryCacheCount = retryCacheCount == null ? new AtomicInteger(0) : retryCacheCount;

            //当前次数
            int currentCount = updateRetryCacheCount.incrementAndGet();

            //失败超过retryCount设定的次数，禁止登录
            if (currentCount > retryCount()) {
                throw new LockedAccountException(String.valueOf(currentCount));
            }

            RedisUtils.setCacheObject(keyName, updateRetryCacheCount, Duration.ofMinutes(retryLockTime()));

            if (currentCount == 3) {
                throw new ExcessiveAttemptsException();
            }
        }
        return matches;
    }

    /**
     * 密码错误重试记录次数
     */
    public int retryCount() {

        return sysProperties.getLogin().getRetryCount();
    }

    /**
     * 密码错误锁定时间
     */
    public int retryLockTime() {

        return sysProperties.getLogin().getRetryLockTime();
    }
}
