package cn.smarthse.core.framework.bootstrap.properties;

import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.NestedConfigurationProperty;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * 系统配置
 *
 * @author liaoly(廖凌云) [1302013247@qq.com]
 * @date 2023/6/20 13:55
 */
@ConfigurationProperties(prefix = "sys")
public @Data class SysProperties {

    /**
     * 大数据日志appcode
     */
    private String bigDataLogAppCode;

    /**
     * 定时任务开启or关闭  on：开启 off：关闭。defauld：off
     */
    private String jobSwitch = "off";

    /**
     * 万达推送地址
     */
    private String wandaPullUrl;
    /**
     * 万达接口加签密钥
     */
    private String wandaAppSecret;

    /**
     * 职业健康/在线培训管理对接密匙
     */
    public String healthOnlineTrainingAesKey;

    /**
     * 在线培训管理地址
     */
    public String healthOnlineTrainingUrl;

    /**
     * 在线培训管理登录后回调地址
     */
    public String healthOnlineTrainingRedirectUrl;

    /**
     * 职业健康在线监管端地址
     */
    public String healthSupUrl;

    /**
     * 证书预览地址QR
     */
    private String certQrUrl;

    /**
     * 文件上传支持的文件类型
     */
    private List<String> supportFileTypes = Arrays.asList("doc", "docx", "xls", "xlsx", "ppt", "pptx", "pdf", "jpg", "jpeg", "png", "txt", "wmv", "mp4", "m4v", "zip", "rar");
    private List<String> supportM3U8FileTypes = Arrays.asList(new String[]{"mp4","m4v"});
    /**
     * 登录配置
     */
    @NestedConfigurationProperty
    private Login login = new Login();

    /**
     * 表字段级修改记录配置
     */
    @NestedConfigurationProperty
    private DataChangeRecorder dataChangeRecorder = new DataChangeRecorder();

    /**
     * 浙政钉单点登录配置
     */
    @NestedConfigurationProperty
    private Zwdd zwdd = new Zwdd();

    /**
     * 浙政钉扫码登录配置
     */
    @NestedConfigurationProperty
    private ZwddQr zwddQr = new ZwddQr();

    /**
     * 浙里办配置
     */
    @NestedConfigurationProperty
    private Zlb zlb = new Zlb();

    /**
     * ai配置
     */
    @NestedConfigurationProperty
    public Ai ai = new Ai();

    /**
     * 蓝牙定位网关中间件 配置
     */
    @NestedConfigurationProperty
    public RwLinkIf3 rwLinkIf3 = new RwLinkIf3();

    /**
     * 登录配置
     */
    @Data
    public static class Login {
        /**
         * token验签密钥
         */
        String secret;
        /**
         * 是否支持多个客户端同时在线 true：支持。 default：true
         */
        boolean multiLogin = true;
        /**
         * 开启登录双因子校验
         */
        boolean twoFactorCheck = false;
        /**
         * 密码错误重试记录次数 default 5 次（超过设定值，需等X分钟后再试，缓存目前设定10分钟）
         */
        Integer retryCount = 5;
        /**
         * 密码错误锁定时间/min default 10 min
         */
        Integer retryLockTime = 10;
    }

    /**
     * 表字段级修改记录配置
     */
    @Data
    public static class DataChangeRecorder {

        boolean enable = false;

        /**
         * 记录修改记录字段配置 key：表名，value：字段（key: 字段名; value: 中文含义）<br/>
         * eg: <code>b_company: {name: 单位名称, credit_code: 统一社会信用代码, area_id: 行政区划}</code>
         */
        Map<String, Map<String, String>> tableColumnMap;

        /**
         * 逻辑删除字段，更新该字段需要标记为删除
         */
        String delColumn = "is_valid";
    }

    /**
     * 浙政钉配置
     */
    @Data
    public static class Zwdd {

        private String accessKey;

        private String secretKey;

        private String domainName;

        private String protocal;

        private Long tenantId;
    }

    /**
     * 浙政钉扫码登录配置
     */
    @Data
    public static class ZwddQr {

        private String clientId;

        private String appKey;

        private String appSecret;

        private String loginDomain;

        private String apiDomain;

        private String protocal;

        private String callBackUrl;

        /**
         * 获取扫码地址 iframe嵌入
         */
        public String getQRCodeUrl() {
            return String.format("%s://%s/oauth2/auth.htm?response_type=code&client_id=%s&redirect_uri=%s&scope=get_user_info&authType=QRCODE&embedMode=true", protocal, loginDomain, clientId, callBackUrl);
            // protocal + "://" + loginDomain + "/oauth2/auth.htm?response_type=code&client_id="+clientId+"&redirect_uri="+callBackUrl+"&scope=get_user_info&authType=QRCODE";
        }
    }

    /**
     * 浙里办配置
     */
    @Data
    public static class Zlb {

        private String appKey;
        private String appSecret;
        private String appId;
        private String host = "https://bcdsg.zj.gov.cn:8443"; // 政务网 https://bcdsg.zj.gov.cn:8443 互联网：https://ibcdsg.zj.gov.cn:8443
    }

    @Data
    public static class Ai {

        /**
         * 智谱ai
         */
        @NestedConfigurationProperty
        public Zhipuai zhipuai;

        /**
         * 讯飞ai
         */
        @NestedConfigurationProperty
        public Xfyun xfyun;

        /**
         * 百度智能云千帆AppBuilder
         */
        @NestedConfigurationProperty
        public Appbuilder appbuilder;

        /**
         * 智谱ai <a href="https://open.bigmodel.cn/">智谱ai开放大模型官网</a>
         */
        @Data
        public static class Zhipuai {
            /**
             * api密钥
             */
            private String apiKey;
            /**
             * Retrieval能力：知识库id
             */
            private String knowledgeId;
        }

        /**
         * 讯飞ai <a href="https://console.xfyun.cn/services/bm35">文档</a>
         */
        @Data
        public static class Xfyun {

            /**
             * 讯飞ai
             */
            private String appId;
            private String apiKey;
            private String apiSecret;
            private String hostUrl = "https://spark-api.xf-yun.com/v3.5/chat";
            /**
             * Retrieval能力：知识库id
             */
            private String knowledgeId;
        }

        /**
         * 百度智能云千帆AppBuilder <a href="https://cloud.baidu.com/doc/AppBuilder/s/Flpv3oxup">文档</a>
         */
        @Data
        public static class Appbuilder {
            private String appId;
            private String secretKey;
        }
    }

    /**
     * 蓝牙定位引擎
     */
    @Data
    public static class RwLinkIf3 {
        private String syncPositionDataUrl;
        private String syncGatewayDataUrl;
    }
}
