package cn.smarthse.core.framework.bean;

import cn.hutool.http.HttpUtil;
import cn.smarthse.common.util.leak.SSRFLeakUtil;
import cn.smarthse.core.framework.utils.FileContentTypeUtils;
import cn.smarthse.core.framework.utils.ServletUtils;
import cn.smarthse.core.framework.utils.StringUtils;
import com.alibaba.fastjson.JSONObject;
import com.aliyun.oss.*;
import com.aliyun.oss.common.comm.Protocol;
import com.aliyun.oss.common.utils.BinaryUtil;
import com.aliyun.oss.model.*;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;

import javax.servlet.http.HttpServletRequest;
import java.io.*;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.spec.X509EncodedKeySpec;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 *
 *  《OSS 客户端配置工具类》
 *
 * @author liaoly(廖凌云) [1302013247@qq.com]
 * @date 2022/3/9 15:56
 */
@Data
@Slf4j
public class OssClientBean {

	private static final String FILE_NAME_POINT_STRING=".";//文件名“点”

	//==================OSS 配置相关==================================
	//oss所在域
    private String local = "D://data/";
	private String domain = "oss-cn-hangzhou.aliyuncs.com";
	private String endpoint = "https://" + domain;
	private String accessKeyId;
	private String accessKeySecret;
	private String bucketName = "yanfa-oss-xxx";//开发环境时的配置

	//是否允许上传到OSS（默认flase)
	private boolean asynch = false;
	//非公开空间生成访问串的有效小时数（按秒记）（默认7200毫秒，2小时）
	private int exptime = 7200;
	//
	private OSSClient client = null;
	//是否公用读：=true时，getUrl不需要生成访问密钥串
	private boolean publicread = true;

	private static final String m3u8_suffix = ".m3u8";
	//=========================相关方法=================================
		
		
		/**
	     * 生成签名URL
	     * @param urlfile OSS短地址
	     */
	    public String getUrl(String urlfile){
	    	return getUrl(urlfile, null, exptime);
	    }

		/**
		 * 生成签名URL
		 * @param urlfile OSS短地址
		 * @param exptime 访问有效期/s
		 */
		public String getUrl(String urlfile, int exptime){
			return getUrl(urlfile, null, exptime);
		}
	    
	    /**
	     * 生成签名URL
	     * @param urlfile		OSS短地址
	     * @param filename		自定义Header头，增加下载文件名
	     * @param exptime		访问有效期/s
	     * @return String
	     */
	    public String getUrl(String urlfile, String filename, int exptime){
			if(isAsynch()) {
				if (StringUtils.isEmpty(urlfile)) {
					return "";
				}

				//去掉参数
				urlfile = urlfile.split("\\?")[0];

				//去掉bucketName、domain
				urlfile = urlfile.replace(String.format("https://%s.%s/", bucketName, domain), "");

				//是否公开读
				if (publicread) {
					StringBuilder url = new StringBuilder();
					url.append("https://").append(bucketName).append(".").append(domain).append("/").append(urlfile);
					if (StringUtils.isNotEmpty(filename)) {
						url.append("?response-content-disposition=attachment;filename=").append(filename);
					}
					return url.toString();
				} else {
					//有效期2小时
					Date exp = DateUtils.addSeconds(new Date(), exptime);
					return generatePresignedUrl(bucketName, filename, urlfile, exp, HttpMethod.GET).toString();
				}
			} else {
				//没有oss同步的话返回本地文件地址
				return ServletUtils.getCtx() + "/" + urlfile;
			}
	    }
		
	   /**
	    * 生成访问地址，增加
	    * @param bucketName
	    * @param attachmentFilename
	    * @param key
	    * @param expiration
	    * @param method
	    * @return
	    * @throws ClientException
	    */
	    private URL generatePresignedUrl(String bucketName, String attachmentFilename, String key, Date expiration, HttpMethod method)
				throws ClientException {
	    	
	    	GeneratePresignedUrlRequest request = new GeneratePresignedUrlRequest(bucketName, key);
			request.setExpiration(expiration);
			request.setMethod(method);
			//自定义头文件 // 要重载的返回请求头。
		    ResponseHeaderOverrides responseHeaders = new ResponseHeaderOverrides();
			if (attachmentFilename != null && !attachmentFilename.equals("")) {
				responseHeaders.setContentDisposition("attachment;filename=\""+attachmentFilename+"\"");
				responseHeaders.setContentEncoding("utf-8");
			}

		    request.setResponseHeaders(responseHeaders);
			if (key.endsWith(m3u8_suffix)) {
				Map<String, String> param = new HashMap<>();
				param.put("x-oss-process", "hls/sign");
				request.setQueryParameter(param);
			}
			//
	    	return getClient().generatePresignedUrl(request);
	    }
	    
	    /**
	     * 生成签名URL
	     * @param urlfile
	     */
	    public String getUrlBybucket(String bucket, String urlfile){
	    	if(StringUtils.isEmpty(urlfile) || StringUtils.isEmpty(bucket)){
	    		return "";
	    	}
	    	//远程连接地址，直接返回
	    	if(urlfile.indexOf("http://")>=0 || urlfile.indexOf("https://")>=0){
	    		return urlfile;
	    	}
	    	
	    	
	    	if(publicread){
	    		StringBuffer url = new StringBuffer();
	    		url.append("https://").append(bucketName).append(".").append(domain).append("/").append(urlfile);
	    		//
	    		return url.toString();
	    	}else{
	    		//有效期1小时
	        	Date exp = DateUtils.addSeconds(new Date(), exptime);
	        	URL url = getClient().generatePresignedUrl(bucketName, urlfile, exp, HttpMethod.GET);
	        	
	        	return url.toString();
	    	}
	    	
	    }
	    
	    /**
	     * 生成带签名的缩略图
	     * @param urlfile
	     * @param styleName
	     * @return
	     */
	    public String getThumbnail(String urlfile, String styleName){
	    	if(StringUtils.isEmpty(urlfile)){
	    		return "";
	    	}
	    	//远程连接地址，直接返回
	    	if(urlfile.indexOf("http://")>=0 || urlfile.indexOf("https://")>=0){
	    		return urlfile;
	    	}

	    	URL url = null;
	    	Date exp = DateUtils.addSeconds(new Date(), exptime);
			urlfile += "@!"+styleName;
			url = getClient().generatePresignedUrl(bucketName, urlfile, exp, HttpMethod.GET);
	    	
	    	//System.out.println(url.toString());
	    	
	    	return url.toString();
	    }
	    
	    /**
	     * 
	     * 上传文件 2015年7月2日14:11:27
		 * @param file      		要上传的文件
		 * @param ossUrl        	服务器要访问的路径名 eg: upload/radiationTraining/common/202212/2212141631340001.jpg
	     * @param fileFullname      文件原名(必须带扩展名) eg：张三头像.jpg
	     * @return					返回文件名，空者返回失败 upload/radiationTraining/common/202212/2212141631340001.jpg
	     * @throws OSSException
	     * @throws ClientException
	     * @throws FileNotFoundException
	     */
	    public String OSSPutObject(File file, String ossUrl, String fileFullname) {
	    	return OSSPutObject(file, ossUrl, fileFullname, null, FileContentTypeUtils.contentType(fileFullname));
	    }

	    /**
	     * 
	     * 上传文件 2015年7月2日14:11:27
	     * 
		 * @param file      		要上传的文件
		 * @param ossUrl        服务器要访问的路径名 eg: upload/radiationTraining/common/202212/2212141631340001.jpg
	     * @param contentType       文件类型
	     * @return					返回文件名，空者返回失败 upload/radiationTraining/common/202212/2212141631340001.jpg
	     * @throws OSSException
	     * @throws ClientException
	     * @throws FileNotFoundException
	     */
	    public String OSSPutObject(File file, String ossUrl, String fileFullname, String contentMD5, String contentType) {
	    	//标识中加入文件名
	    	ObjectMetadata objectMeta = new ObjectMetadata();
	    	objectMeta.setContentLength(file.length());
			objectMeta.setContentEncoding("UTF-8");
	    	//文件名
	    	if(StringUtils.isNotEmpty(fileFullname)) {
				try {
					fileFullname = URLEncoder.encode(fileFullname , "UTF-8");
				} catch (UnsupportedEncodingException e) {
					log.error("UnsupportedEncodingException：", e);
				}
				objectMeta.setContentDisposition("inline;filename*=utf-8'zh_cn'" + fileFullname );
				objectMeta.setHeader("Content-disposition", "filename*=utf-8'zh_cn'" + fileFullname );
	    	}
	    	//文件类型
	    	if(StringUtils.isNotEmpty(contentType)) {
	    		objectMeta.setContentType(contentType + ";charset=utf-8");
	    	}
	    	
	    	//文件md5
	    	if(StringUtils.isNotEmpty(contentMD5)) {
	    		objectMeta.setContentMD5(contentMD5);
	    	}

	    	return OSSPutObject(file, ossUrl, objectMeta);
	    }
	    
	    /**
	     * 
	     * 上传文件 2015年7月2日14:11:27
	     * @param file      		要上传的文件
	     * @param ossUrl        	服务器要访问的路径名 eg: upload/radiationTraining/common/202212/2212141631340001.jpg
	     * @param objectMeta        文件元信息
	     * @return					返回文件名，空者返回失败 upload/radiationTraining/common/202212/2212141631340001.jpg
	     * @throws OSSException
	     * @throws ClientException
	     * @throws FileNotFoundException
	     */
	    public String OSSPutObject(File file, String ossUrl, ObjectMetadata objectMeta) {

	    	InputStream input = null;
	    	try {
	    		input = new FileInputStream(file);

				//key eg：upload/radiationTraining/common/202212/2212141631340001.jpg
				ossUrl = ossUrl.replace("\\", "/");

	    		PutObjectResult resultObject = getClient().putObject(bucketName, ossUrl, input, objectMeta);
	    	} catch (FileNotFoundException e) {
				log.error("FileNotFoundException：", e);
	    		return "";
	    	} finally {
				try {
					input.close();
				} catch (IOException e) {
					log.error("IOException：", e);
				}
			}
			return ossUrl;
	    }

	public String OSSPutObject(String key,String fileFullname, InputStream inputStream) throws IOException {
		//标识中加入文件名
		ObjectMetadata objectMeta = new ObjectMetadata();
		objectMeta.setContentLength(inputStream.available());
		objectMeta.setContentEncoding("UTF-8");
		//文件名
		if (StringUtils.isNotEmpty(fileFullname)) {
			try {
				fileFullname = URLEncoder.encode(fileFullname, "UTF-8");
			} catch (UnsupportedEncodingException e) {
				log.error("UnsupportedEncodingException：", e);
			}
			objectMeta.setContentDisposition("inline;filename*=utf-8'zh_cn'" + fileFullname);
			objectMeta.setHeader("Content-disposition", "filename*=utf-8'zh_cn'" + fileFullname);
		}
		//文件类型
		String contentType = FileContentTypeUtils.contentType(fileFullname);
		if (StringUtils.isNotEmpty(contentType)) {
			objectMeta.setContentType(contentType + ";charset=utf-8");
		}

		//文件md5
		objectMeta.setContentMD5(null);

		return OSSPutObject(key, inputStream, null);
	}

	public String OSSPutObject(String key, InputStream inputStream, ObjectMetadata objectMeta) {
		PutObjectResult putObjectResult;
		try {
			key = key.replace("\\", "/");
			putObjectResult = getClient().putObject(
					bucketName,
					key,
					inputStream,objectMeta);
		} catch (OSSException e) {
			throw new RuntimeException(e);
		} catch (ClientException e) {
			throw new RuntimeException(e);
		} finally {
			try {
				inputStream.close();
			} catch (IOException e) {
				log.error("IOException：", e);
			}
		}
		return key;
	}

	    /**
	     * 下载文件  
	     * @param file				File.ossurl
	     * @return
	     * @throws OSSException
	     * @throws ClientException
	     */
	    public OSSObject  downloadFile( String file) {  
	    	//fileclient.getObject(new GetObjectRequest(OSSFILE, file), new File(filename));  
	    	OSSObject obj=null;
	    	try{
	    	obj=getClient().getObject(bucketName, file);
	    	} catch(OSSException oe){
	    		log.error("阿里云文件获取失败:{}", oe.getMessage());
	    	} catch (ClientException ce){
				log.error("阿里云连接失败:{}", ce.getMessage());
	    	}catch(NullPointerException n){
				log.error("阿里云文件获取失败:{}", n.getMessage());
	    	}
	    	
	    	return obj;
	    }
	    
	    /**
	     * 从阿里云下载文件
	     * @param fileOssurl 源文件相对路径（文件在阿里云的相对路径，也是File实体类的ossurl属性）
	     * @param destFilePath 需要生成的目标文件路径
	     * @param fileName 需要生成的目标文件名称
	     * @return 文件名（如果有重复文件名，则返回重命名后的文件名；如果没有，则返回原文件名）
	     * @throws IOException
	     */
	    public String downloadFile(String fileOssurl,String destFilePath,String fileName) throws IOException{
	    	OSSObject ossObject=this.downloadFile(fileOssurl);
	    	if(StringUtils.isEmpty(fileOssurl)|| StringUtils.isEmpty(destFilePath)||ossObject==null||ossObject.getObjectContent()==null){
	    		return fileName;
	    	}
	    	BufferedOutputStream bos = null;
	    	InputStream reader = null;
	    	String[] names={fileName};
	    	
	    	File file = this.getNoRepeatDestFilePath(destFilePath, names);
	    	File parentFile = file.getParentFile();
	        if (parentFile != null) {
	            if (!parentFile.mkdirs() && !parentFile.isDirectory()) {
	                throw new IOException("目录创建失败");
	            }
	        }
	    	
	    	try {
				bos = new BufferedOutputStream(new FileOutputStream(file));
				reader = ossObject.getObjectContent();
				int len = 2048;
	            byte[] b = new byte[len];
	            while ((len = reader.read(b)) > 0) {
					bos.write(b, 0, len);
	            }
	            bos.flush();
			} catch (FileNotFoundException e) {
				log.error("文件未找到:{}",  e.getMessage());
			} catch (IOException e) {
				log.error("文件下载失败，输入输出错误:{}", e.getMessage());
			}finally {
	            if (reader != null) {
	                reader.close();
	            }
	            if (bos != null) {
					bos.close();
				}
	        }
	    	return names[0];
	    }

	public InputStream getInputStreamByUrl(String fileUrl) {
		InputStream reader;
		try {
			if (StringUtils.isEmpty(fileUrl)) {
				return null;
			}
			//ssrf漏洞
			//限制协议，仅允许http或https协议
			if (!(fileUrl.startsWith("http") || fileUrl.startsWith("https"))) {
				log.error("该请求不是以http或https开头");
				return null;
			}
			//拦截返回文件是否符合规定
			if (!SSRFLeakUtil.securitySSRFFileTypeCheck(null, fileUrl)) {
				return null;
			}
			cn.hutool.http.HttpResponse execute = HttpUtil.createGet(fileUrl).execute();
			reader = execute.bodyStream();
			return reader;
		} catch (IOException e) {
			e.printStackTrace();
			return null;
		}
	}

	    /**
	     * 获取目标文件对象（检查重复，不覆盖）
	     * @param destFilePath 目标路径
	     * @param fileName 文件名
	     * @return 目标文件对象
	     */
	    private File getNoRepeatDestFilePath(String destFilePath,String[] fileName){
	    	if (!destFilePath.endsWith(File.separator)){
	    		destFilePath = destFilePath+File.separator;
	    	}
	    	String fileSuffix = "";
	    	String pointStr = FILE_NAME_POINT_STRING;
	    	int pointIndex = fileName[0].lastIndexOf(FILE_NAME_POINT_STRING);
	    	if (pointIndex > 0){
	    		fileSuffix = fileName[0].substring(pointIndex+1);
	    		fileName[0] = fileName[0].substring(0, pointIndex);
	    	} else {
	    		pointStr="";
	    	}
	    	
	    	File file = new File(destFilePath+fileName[0]+pointStr+fileSuffix);
	    	int i=1;
	    	while (file.exists()){
	    		file=new File(destFilePath+fileName[0]+i+pointStr+fileSuffix);
	    		i++;
	    	}
	    	if (i>1) {
	    		fileName[0]=fileName[0]+--i+pointStr+fileSuffix;
	    	} else {
	    		fileName[0]=fileName[0]+pointStr+fileSuffix;
	    	}
	    	
	    	return file;
	    }
	    
	    /**
	     * 读取文件元信息
	     * @param url
	     * @return
	     */
	    public ObjectMetadata getObjectMetadataByUrl(String url){

	    	return getClient().getObjectMetadata(bucketName, url);
	    }

	    /**
	     * 生成企业+用户 直传OSS签名
	     * 
	     * @Comments:  <对此方法的描述，可以引用系统设计中的描述>
	     * @author JannyShao(邵建义) [ksgameboy@qq.com]
	     * @since 2019年2月25日-下午2:06:45
	     * @param dir		上传目录
	     * @return
	     * @throws Exception
	     */
	    public Map<String, String> getPostSignature(String dir) throws Exception {
	    	//
	    	String host = "https://" + bucketName + "." + domain; // host的格式为 bucketname.endpoint
			//
	    	long expireTime = 30;
			long expireEndTime = System.currentTimeMillis() + expireTime * 1000;
			Date expiration = new Date(expireEndTime);
			PolicyConditions policyConds = new PolicyConditions();
			policyConds.addConditionItem(PolicyConditions.COND_CONTENT_LENGTH_RANGE, 0, 1048576000);
			policyConds.addConditionItem(MatchMode.StartWith, PolicyConditions.COND_KEY, dir);

			String postPolicy = getClient().generatePostPolicy(expiration, policyConds);
			byte[] binaryData = postPolicy.getBytes("utf-8");
			String encodedPolicy = BinaryUtil.toBase64String(binaryData);
			String postSignature = getClient().calculatePostSignature(postPolicy);
			
			Map<String, String> respMap = new LinkedHashMap<String, String>();
			respMap.put("accessid", accessKeyId);
			respMap.put("policy", encodedPolicy);
			respMap.put("signature", postSignature);
			respMap.put("dir", dir);
			respMap.put("host", host);
			respMap.put("expire", String.valueOf(expireEndTime / 1000));
			
			JSONObject jasonCallback = new JSONObject();
//			jasonCallback.put("callbackUrl", callbackUrl);
			//自定义参数示例:
//			String callbackbody = "{\'bucket\':${bucket}, \'object\':${object},\'size\':${size},\'mimeType\':${mimeType},\'uid\':${x:uid},\'biz_type\':${x:biz_type}}";
//			if("1".equals(biz_type)){//用户头像，banner修改
//			callbackbody = "{\'bucket\':${bucket}, \'object\':${object},\'size\':${size},\'mimeType\':${mimeType},\'uid\':${x:uid},\'biz_type\':${x:biz_type},\'portrait\':${x:portrait},\'banner\':${x:banner}}";
//			}else if("2".equals(biz_type)){//投诉建议
//			callbackbody = "{\'bucket\':${bucket}, \'object\':${object},\'size\':${size},\'mimeType\':${mimeType},\'uid\':${x:uid},\'biz_type\':${x:biz_type},\'path\':${x:path},\'guideid\':${x:guideid}}";
//			}
			StringBuffer callbackBody = new StringBuffer("{");
			callbackBody.append("bucket:").append("${bucket}");
			callbackBody.append(",filename:").append("${object}");
			callbackBody.append(",etag:").append("${etag}");
			callbackBody.append(",size:").append("${size}");
			callbackBody.append(",mimeType:").append("${mimeType}");
			callbackBody.append(",height:").append("${imageInfo.height}");
			callbackBody.append(",width:").append("${imageInfo.width}");
			callbackBody.append(",format:").append("${imageInfo.format}");
			//自定义, 当前上传企业 + 用户Id
//			callbackBody.append(",uid:").append(uid);
//			callbackBody.append(",cid:").append(cid);
//			callbackBody.append(",description:\'").append(description).append("\'");
			callbackBody.append("}");
			jasonCallback.put("callbackBody",
//					"filename=${object}&size=${size}&mimeType=${mimeType}&height=${imageInfo.height}&width=${imageInfo.width}"
					callbackBody.toString()
					);
			//bucket=${bucket}&object=${object}&etag=${etag}&size=${size}&mimeType=${mimeType}&imageInfo.height=${imageInfo.height}&imageInfo.width=${imageInfo.width}&imageInfo.format=${imageInfo.format}&my_var=${x:my_var}
			jasonCallback.put("callbackBodyType", "application/json");
//			jasonCallback.put("callbackHost", "可选参数，默认为callbackUrl");
//			jasonCallback.put("callbackBodyType", "可选参数，支持application/x-www-form-urlencoded和application/json，默认为前者");
			
			
			String base64CallbackBody = BinaryUtil.toBase64String(jasonCallback.toString().getBytes());
			respMap.put("callback", base64CallbackBody);
			return respMap;
	    }
	    
	    /**
		 * 验证上传回调的Request
		 * 
		 * @param request
		 * @param ossCallbackBody
		 * @return
		 * @throws NumberFormatException
		 * @throws IOException
		 */
	    public boolean VerifyOSSCallbackRequest(HttpServletRequest request, String ossCallbackBody)
				throws NumberFormatException, IOException {
			boolean ret = false;
			String autorizationInput = new String(request.getHeader("Authorization"));
			String pubKeyInput = request.getHeader("x-oss-pub-key-url");
			byte[] authorization = BinaryUtil.fromBase64String(autorizationInput);
			byte[] pubKey = BinaryUtil.fromBase64String(pubKeyInput);
			String pubKeyAddr = new String(pubKey);
			if (!pubKeyAddr.startsWith("http://gosspublic.alicdn.com/")
					&& !pubKeyAddr.startsWith("https://gosspublic.alicdn.com/")) {
				System.out.println("pub key addr must be oss addrss");
				return false;
			}
			String retString = executeGet(pubKeyAddr);
			retString = retString.replace("-----BEGIN PUBLIC KEY-----", "");
			retString = retString.replace("-----END PUBLIC KEY-----", "");
			String queryString = request.getQueryString();
			String uri = request.getRequestURI();
			String decodeUri = java.net.URLDecoder.decode(uri, "UTF-8");
			String authStr = decodeUri;
			if (queryString != null && !queryString.equals("")) {
				authStr += "?" + queryString;
			}
			authStr += "\n" + ossCallbackBody;
			ret = doCheck(authStr, authorization, retString);
			return ret;
		}
	    
	    
	    /**
		 * 获取public key
		 * 
		 * @param url
		 * @return
		 */
		@SuppressWarnings({ "finally" })
		public String executeGet(String url) {
			BufferedReader in = null;

			String content = null;
			try {
				// 定义HttpClient
				@SuppressWarnings("resource")
				DefaultHttpClient client = new DefaultHttpClient();
				// 实例化HTTP方法
				HttpGet request = new HttpGet();
				request.setURI(new URI(url));
				HttpResponse response = client.execute(request);

				in = new BufferedReader(new InputStreamReader(response.getEntity().getContent()));
				StringBuffer sb = new StringBuffer("");
				String line = "";
				String NL = System.getProperty("line.separator");
				while ((line = in.readLine()) != null) {
					sb.append(line + NL);
				}
				in.close();
				content = sb.toString();
			} catch (Exception e) {
			} finally {
				if (in != null) {
					try {
						in.close();// 最后要关闭BufferedReader
					} catch (Exception e) {
						log.error("Exception：", e);
					}
				}
				return content;
			}
		}
	    
		/**
		 * 验证RSA
		 * 
		 * @param content
		 * @param sign
		 * @param publicKey
		 * @return
		 */
		public static boolean doCheck(String content, byte[] sign, String publicKey) {
			try {
				KeyFactory keyFactory = KeyFactory.getInstance("RSA");
				byte[] encodedKey = BinaryUtil.fromBase64String(publicKey);
				PublicKey pubKey = keyFactory.generatePublic(new X509EncodedKeySpec(encodedKey));
				java.security.Signature signature = java.security.Signature.getInstance("MD5withRSA");
				signature.initVerify(pubKey);
				signature.update(content.getBytes());
				boolean bverify = signature.verify(sign);
				return bverify;

			} catch (Exception e) {
				log.error("Exception：", e);
			}

			return false;
		} 
	    
	    /**
		 * 生成host全路径
		 * @return
		 */
	    public String getHost(){
	    	return "https://" + bucketName + "." + domain;
	    }

		/**
		 * @return the client
		 */
		public OSSClient getClient() {

			if(client == null) {

				ClientConfiguration configuration = new ClientConfiguration();
				configuration.setSupportCname(false);
				if (endpoint != null && endpoint.contains(Protocol.HTTPS.toString())) {
					configuration.setProtocol(Protocol.HTTPS);
				}

				client = new OSSClient(endpoint,accessKeyId, accessKeySecret, configuration);
			}

			return client;
		}
}
