package cn.smarthse.core.framework.aspect;

import cn.smarthse.common.redis.utils.RedisUtils;
import cn.smarthse.core.framework.annotation.RequestLimit;
import cn.smarthse.core.framework.model.ResponseStateEnum;
import cn.smarthse.core.framework.utils.StringUtils;
import cn.smarthse.core.framework.utils.login.JwtUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.support.StandardMultipartHttpServletRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.smarthse.framework.exception.ServiceException;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.nio.charset.StandardCharsets;
import java.time.Duration;

/**
 * @author haosw
 * @Description 日志切面类
 * @date 2021/4/29
 */
@Component
@Aspect
@RequiredArgsConstructor
@Slf4j
public class RequestLimitAspect {

    /**
     * 2.创建增强 返回值通知
     */
    @Before("@annotation(cn.smarthse.core.framework.annotation.RequestLimit)")
    public void logBefor(JoinPoint joinPoint) {

        //通过获取当前方法上的注解 得到日志的内容
        //获取方法签名对象
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Object[] args = joinPoint.getArgs();

        //获得当前方法对象
        Method method = signature.getMethod();
        // 获取方法中是否包含注解
        RequestLimit anno = method.getAnnotation(RequestLimit.class);

        if (anno != null) {
            //获取request
            HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();

            //解析并执行spel表达式
            String spelValue = invokeExpression(method, args, anno.expression());
            if (isLimit(request, anno, spelValue)) {
                // 返回请求限制错误
                throw new ServiceException(ResponseStateEnum.RequestLimit.getValue(), anno.msg(), null);
            }
        }
    }

    /**
     * 将字符串渲染到客户端
     *
     * @param response 渲染对象
     * @param string   待渲染的字符串
     */
    public static void renderString(HttpServletResponse response, String string) {
        response.setStatus(HttpStatus.OK.value());
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.setContentType("application/json;charset=UTF-8");
        try (PrintWriter writer = response.getWriter()) {
            writer.print(string);
        } catch (IOException e) {
            // ignore
        }
    }

    /**
     * 判断是否超过次数限制
     *
     * @param request           HttpServletRequest
     * @param RepeatUrlFormData RepeatUrlFormData
     * @return boolean true表示超过
     */
    public boolean isLimit(HttpServletRequest request, RequestLimit RepeatUrlFormData, String param) {
        if (request instanceof StandardMultipartHttpServletRequest) {
            return false;
        }

        Long userId = JwtUtil.getUserId();        //userId 	  	eg: 10132506028412922
        String servletPath = request.getServletPath();  //接口路径 		eg: /admin/person_user/importPersonUser
        String limitKey = String.format("LIMIT:%s:%s:%s", userId, servletPath, param);

        Integer count = RedisUtils.getCacheObject(limitKey);
        if (count == null) {
            // 初始化次数
            RedisUtils.setCacheObject(limitKey, 1, Duration.ofSeconds(RepeatUrlFormData.second()));
        } else {
            if (count >= RepeatUrlFormData.maxCount()) {
                return true;
            }
            // 次数自增
            RedisUtils.setCacheObject(limitKey, count + 1, Duration.ofSeconds(RepeatUrlFormData.second()));
        }
        return false;
    }

    /**
     * 解析并执行spel表达式
     *
     * @param method
     * @param args
     * @param expression spel表达式
     * @return {@link String}
     * @author liaoly(廖凌云)
     * @date 2023/11/9 15:34
     */
    private String invokeExpression(Method method, Object[] args, String expression) {

        if (StringUtils.isEmpty(expression)) {
            return null;
        }

        EvaluationContext context = new StandardEvaluationContext();

        //获取运行时参数的名称
        LocalVariableTableParameterNameDiscoverer discoverer = new LocalVariableTableParameterNameDiscoverer();
        String[] parameterNames = discoverer.getParameterNames(method);

        //将参数绑定到context中
        try {
            if (parameterNames != null) {
                for (int i = 0; i < parameterNames.length; i++) {
                    context.setVariable(parameterNames[i], args[i]);
                }
            }
        } catch (Exception e) {
            log.error("invokeExpression：", e);
        }

        //解析并执行spel表达式
        return String.valueOf(new SpelExpressionParser().parseExpression(expression).getValue(context));
    }
}
