package cn.smarthse.rho.modules.sys.utils;


import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.IdcardUtil;
import cn.hutool.core.util.ReUtil;
import cn.hutool.core.util.StrUtil;
import cn.smarthse.rho.modules.sys.enums.staff.IdCardTypeEnum;

import java.time.LocalDate;
import java.time.Period;

public class RhoIdCardUtil {

    private static final String HK_MC_HOME_RETURN_PERMIT_REGEX = "^[HM]\\d{8}$";
    private static final String TW_HOME_RETURN_PERMIT_REGEX = "^\\d{8}$";

    public static boolean isValidIdCard_15_18(String idCard){
        return IdcardUtil.isValidCard18(idCard) || IdcardUtil.isValidCard15(idCard);
    }


    /**
     * 判断是否是大陆地区身份证
     * 注意：身份证只校验18位，系统不允许存在15位身份证，如果有15位校验需求，请直调用其他方法，或者重新起一个方法
     * @param idCard
     * @return
     */
    public static boolean isValid_CN_DL_IdCard18(String idCard) {
        if(StrUtil.isBlank(idCard)){
            return false;
        }
        String idCardType = getIdCardType(idCard);
        return  IdcardUtil.isValidCard18(idCard) && IdCardTypeEnum.ID_CARD.getValue().equals(idCardType) ;
    }

    /**
     * 判断是否是港澳台居留证
     * @param idCard
     * @return
     */
    public static boolean isValid_CN_HkMcTw_IdCard(String idCard) {
        if(StrUtil.isBlank(idCard)){
            return false;
        }
        String idCardType = getIdCardType(idCard);
        return  IdCardTypeEnum.ID_CARD_HK_MC_TW.getValue().equals(idCardType) ;
    }


    /**
     * 外国人永久居留证
     * @param idCard
     * @return
     */
    public static boolean isValid_CN_Wxk_IdCard(String idCard) {
        if(StrUtil.isBlank(idCard)){
            return false;
        }
        String idCardType = getIdCardType(idCard);
        return  IdCardTypeEnum.ID_CARD_WXK.getValue().equals(idCardType) ;
    }


    /**
     * 是否是港澳来往通行证
     * @param idCard
     * @return
     */
    public static boolean isValid_HkMc_RoundTripPermit(String idCard){
        if(StrUtil.isBlank(idCard)){
            return false;
        }
        return ReUtil.isMatch(HK_MC_HOME_RETURN_PERMIT_REGEX, idCard);
    }

    /**
     * 是否是台湾来往通行证
     * @param idCard
     * @return
     */
    public static boolean isValid_Tw_RoundTripPermit(String idCard){
        if(StrUtil.isBlank(idCard)){
            return false;
        }
        return ReUtil.isMatch(TW_HOME_RETURN_PERMIT_REGEX, idCard);
    }

    /**
     * 识别证件号类型
     * @param idCard
     * @return
     */
    public static String getIdCardType(String idCard){
        if(StrUtil.isBlank(idCard)){
            return null;
        }

        //18位的
        if(cn.hutool.core.util.IdcardUtil.isValidCard18(idCard)){
            //香港81 澳门82 台湾83
            if(idCard.startsWith("81") || idCard.startsWith("82") || idCard.startsWith("83")){
                return IdCardTypeEnum.ID_CARD_HK_MC_TW.getValue();
            }else if(idCard.startsWith("9")){
                //外国人永久居留证
                return IdCardTypeEnum.ID_CARD_WXK.getValue();
            }else {
                return IdCardTypeEnum.ID_CARD.getValue();
            }
        }

        //15位的
        if(cn.hutool.core.util.IdcardUtil.isValidCard15(idCard)){
            return IdCardTypeEnum.ID_CARD_HK_MC_TW.getValue();
        }

        //港澳来往通行证
        if(RhoIdCardUtil.isValid_HkMc_RoundTripPermit(idCard)){
            return IdCardTypeEnum.HK_MC_HOME_RETURN_PERMIT.getValue();
        }

        //台湾来往通行证
        if(RhoIdCardUtil.isValid_Tw_RoundTripPermit(idCard)){
            return IdCardTypeEnum.TW_HOME_RETURN_PERMIT.getValue();
        }


        return null;

    }



    public static Integer getSex(String idCard, Integer ifNullResult){
        return isValidIdCard_15_18(idCard) ? Integer.valueOf(IdcardUtil.getGenderByIdCard(idCard)) : ifNullResult;
    }

    public static Integer getAge(String idCard, Integer ifNullResult){
        return isValidIdCard_15_18(idCard) ? Integer.valueOf(IdcardUtil.getAgeByIdCard(idCard)) : ifNullResult;
    }

    public static Integer getAge(LocalDate birthDate){
        return Period.between(birthDate, LocalDate.now()).getYears();
    }

    /**
     * 获取年龄，如果证件号获取不了，就通过出生日期获取，如果还获取不了就返回ifNullResult
     * @param idCard 证件号
     * @param birthDate 出生日期
     * @param ifNullResult 如果空返回的值
     * @return
     */

    public static Integer getAge(String idCard, LocalDate birthDate ,Integer ifNullResult){
        if(isValidIdCard_15_18(idCard)){
            return Integer.valueOf(IdcardUtil.getAgeByIdCard(idCard));
        }else {
            if(birthDate != null){
                return getAge(birthDate);
            }
        }
        return ifNullResult;
    }

    public static LocalDate getBirth(String idCard, LocalDate ifNullResult){
        return isValidIdCard_15_18(idCard) ? LocalDate.from(DateUtil.toLocalDateTime(IdcardUtil.getBirthDate(idCard))) : ifNullResult;
    }
}
