package cn.smarthse.rho.modules.sys.utils;

import cn.hutool.core.util.StrUtil;

import java.util.HashMap;
import java.util.Map;

/**
 * @author chuhy
 * @date 2025/5/22 13:54
 */
public class NeutronNameUtil {

    /**
     * 系统原格式（上标）名称转换为拼接格式名称
     * 例：⁴²Mo → 42-Mo
     * @param original
     * @return {@link String}
     * @author chuhy
     * @date 2025/5/22 14:07
     */
    public static String originalName2MontageName(String original) {
        // 创建上标字符到普通数字的映射
        Map<Character, Character> superscriptMap = new HashMap<>();
        superscriptMap.put('⁰', '0');
        superscriptMap.put('¹', '1');
        superscriptMap.put('²', '2');
        superscriptMap.put('³', '3');
        superscriptMap.put('⁴', '4');
        superscriptMap.put('⁵', '5');
        superscriptMap.put('⁶', '6');
        superscriptMap.put('⁷', '7');
        superscriptMap.put('⁸', '8');
        superscriptMap.put('⁹', '9');

        // 分割中子数和元素符号
        int splitIndex = 0;
        while (splitIndex < original.length() && superscriptMap.containsKey(original.charAt(splitIndex))) {
            splitIndex++;
        }

        String neutronPart = original.substring(0, splitIndex);
        String elementPart = original.substring(splitIndex);

        // 转换中子数部分
        StringBuilder convertedNeutron = new StringBuilder();
        for (char c : neutronPart.toCharArray()) {
            convertedNeutron.append(superscriptMap.get(c));
        }

        // 组合最终结果
        return convertedNeutron.toString() + "-" + elementPart;
    }


    /**
     * 拼接格式转化为上标格式
     * 例：42-Mo → ⁴²Mo
     * @param original
     * @return {@link String}
     * @author chuhy
     * @date 2025/5/22 14:07
     */
    public static String montageName2OriginalName(String original) {
        // 创建普通数字到上标字符的映射
        Map<Character, Character> subscriptMap = new HashMap<>();
        subscriptMap.put('0', '⁰');
        subscriptMap.put('1', '¹');
        subscriptMap.put('2', '²');
        subscriptMap.put('3', '³');
        subscriptMap.put('4', '⁴');
        subscriptMap.put('5', '⁵');
        subscriptMap.put('6', '⁶');
        subscriptMap.put('7', '⁷');
        subscriptMap.put('8', '⁸');
        subscriptMap.put('9', '⁹');

        // 分割数字部分和元素符号
        String[] parts = original.split("-", 2); // 最多分割成两部分
        if (parts.length != 2) {
            // 如果格式不合法，直接返回原字符串（或抛异常）
            return original;
        }
        String numberPart = parts[0];
        String elementPart = parts[1];

        // 转换数字部分为上标
        StringBuilder convertedNumber = new StringBuilder();
        for (char c : numberPart.toCharArray()) {
            Character supChar = subscriptMap.get(c);
            if (supChar != null) {
                convertedNumber.append(supChar);
            } else {
                // 如果遇到非数字字符，保留原字符（可根据需求调整）
                convertedNumber.append(c);
            }
        }

        // 组合最终结果
        return convertedNumber.toString() + elementPart;
    }


    /**
     * 校验对方的名称格式
     * @param input
     * @return {@link boolean}
     * @author chuhy
     * @date 2025/5/22 15:49
     */
    public static boolean isValidMontageName(String input) {

        // 规则1: 必须包含且仅包含一个连字符
        if (input.chars().filter(c -> c == '-').count() != 1) {
            // 必须包含且仅包含一个连字符 '-'
            return false;
        }

        // 规则2: 分割后的两部分必须非空
        String[] parts = input.split("-", 2);
        if (parts[0].isEmpty()) {
            // 连字符前必须有质量数
            return false;
        }
        if (parts[1].isEmpty()) {
            // 连字符后必须有元素符号
            return false;
        }

        // 规则3: 质量数必须为纯数字
        if (!parts[0].matches("\\d+")) {
            // 质量数包含非数字字符
            return false;
        }

        // 规则4: 元素符号必须为纯字母（支持大小写）
        if (!parts[1].matches("[A-Za-z]+")) {
            // 元素符号包含非法字符
            return false;
        }

        return true;
    }


    /**
     * 格式化名称
     *
     * @param symbol
     * @param neutrons
     * @param energyState
     * @return {@link String}
     * @author chuhy
     * @date 2025/7/8 14:30
     */
    public static String formatName(String symbol, String neutrons, String energyState) {
        // 将neutrons中普通的数字和字符替换成上标数字和字符
        Map<Character, Character> superscriptMap = new HashMap<>();
        superscriptMap.put('0', '⁰');
        superscriptMap.put('1', '¹');
        superscriptMap.put('2', '²');
        superscriptMap.put('3', '³');
        superscriptMap.put('4', '⁴');
        superscriptMap.put('5', '⁵');
        superscriptMap.put('6', '⁶');
        superscriptMap.put('7', '⁷');
        superscriptMap.put('8', '⁸');
        superscriptMap.put('9', '⁹');
        superscriptMap.put('m', 'ᵐ');

        StringBuilder convertedNeutrons = new StringBuilder();
        for (char c : neutrons.toCharArray()) {
            Character supChar = superscriptMap.get(c);
            if (supChar != null) {
                convertedNeutrons.append(supChar);
            } else {
                convertedNeutrons.append(c);
            }
        }
        if (!StrUtil.isBlank(energyState)){
            energyState = "(" + energyState + ")";
        }
        return convertedNeutrons + symbol + energyState;

    }

}
