package cn.smarthse.radiationTraining.core.framework.utils;

import cn.hutool.core.util.StrUtil;
import cn.smarthse.framework.core.utils.ServletUtils;
import cn.smarthse.radiationTraining.core.framework.Constant;
import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import jakarta.servlet.http.HttpServletRequest;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import java.util.Date;

/**
 * @Author liaoly(廖凌云) [1302013247@qq.com]
 * @Date 2022/2/20 14:16
 */
@Component
public class JwtUtil {

	private static String audience;

	// 当Spring创建JwtUtil的实例时，会调用setter方法，从而设置静态变量
	@Value("${jwt.audience}")
	public void setAudience(String audience) {
		JwtUtil.audience = audience;
	}

	/**
	 * 获得token中的信息无需secret解密也能获得
	 * @param token token
	 * @return token中包含的用户手机号or账号
	 */
	public static String getUsername(String token) {
		try {
			DecodedJWT jwt = JWT.decode(token);
			return jwt.getClaim("username").asString();
		}
		catch (JWTDecodeException e) {
			return null;
		}
	}

	/**
	 * 获得token中的信息无需secret解密也能获得
	 * @param token token
	 * @return token中包含的用户id
	 */
	public static Long getUserId(String token) {
		try {
			DecodedJWT jwt = JWT.decode(token);
			return jwt.getClaim("userId").asLong();
		}
		catch (JWTDecodeException e) {
			return null;
		}
	}

	/**
	 * 获得token中的信息无需secret解密也能获得
	 * @param token token
	 * @return token中包含的用户姓名
	 */
	public static String getName(String token) {
		try {
			DecodedJWT jwt = JWT.decode(token);
			return jwt.getClaim("name").asString();
		}
		catch (JWTDecodeException e) {
			return null;
		}
	}

	/**
	 * 获得token中的信息无需secret解密也能获得
	 * @param token token
	 * @return token中包含的ip
	 */
	public static String getIP(String token) {
		try {
			DecodedJWT jwt = JWT.decode(token);
			return jwt.getClaim("ip").asString();
		}
		catch (JWTDecodeException e) {
			return null;
		}
	}

	/**
	 * 获取当前登录用户的token,如果 HttpServletRequest 为空的话，尝试从 MDC中获取；token为null则获取refreshToken
	 *
	 * @return token
	 */
	public static String getToken() {

		HttpServletRequest request = ServletUtils.getRequest();
		String token;
		if (request == null) {
			token = MDC.get(Constant.TOKEN_HEADER_NAME);
		} else {
			token = request.getHeader(Constant.TOKEN_HEADER_NAME);
			if (StrUtil.isBlank(token)) {
				token = request.getParameter("refreshToken");
			}
		}
		return token;
	}

	/**
	 * 获取当前登录用户的token,如果token为null则获取refreshToken
	 * @param request HttpServletRequest
	 * @return token
	 */
	public static String getToken(HttpServletRequest request) {
		String token = request.getHeader(Constant.TOKEN_HEADER_NAME);
		if (StrUtil.isBlank(token)) {
			return request.getParameter("refreshToken");
		} else {
			return token;
		}
	}

	/**
	 * 创建token没有携带ip
	 * @param username 用户名/手机号
	 * @param userId 用户id
	 * @param secret 用户的密码
	 * @param time token的有效时间 单位:毫秒
	 * @return 加密的token
	 */
	public static String createToken(String username, Long userId, String secret, Long time) {
		Date date = new Date(System.currentTimeMillis() + time);
		Algorithm algorithm = Algorithm.HMAC256(secret);
		return JWT.create()
				.withClaim("username", username)
				.withClaim("userId", userId)
				.withAudience(audience)
				.withExpiresAt(date).sign(algorithm);
	}

	/**
	 * 创建token携带ip
	 * @param username 用户名/手机号
	 * @param userId 用户id
	 * @param ip 用户当前ip
	 * @param secret 用户的密码(token解密密钥)
	 * @param time token的有效时间 单位:毫秒
	 * @return 加密的token
	 */
	public static String createToken(String username, Long userId, String name, String ip, String secret, Long time) {
		Date date = new Date(System.currentTimeMillis() + time);
		Algorithm algorithm = Algorithm.HMAC256(secret);
		return JWT.create()
				.withClaim("username", username)
				.withClaim("userId", userId)
				.withClaim("name", name)
				.withClaim("ip", ip)
				.withAudience(audience)
				.withExpiresAt(date).sign(algorithm);
	}

	/**
	 * 校验token是否合法
	 * @param token 密钥
	 * @param secret 用户的密码(token解密密钥)
	 * @return 是否合法 合法返回true
	 */
	public static boolean verify(String token, String secret) {

		if (StringUtils.isBlank(secret)) {
			return false;
		}
		try {
			// 根据密码生成JWT效验器
			Algorithm algorithm = Algorithm.HMAC256(secret);
			JWTVerifier verifier = JWT.require(algorithm).withClaim("username", getUsername(token))
					.withClaim("userId", getUserId(token))
					.withAudience(audience)
					.build();
			// 效验TOKEN
			verifier.verify(token);
			return true;
		}
		catch (JWTVerificationException exception) {
			return false;
		}
	}

	/**
	 * 校验token是否正确（携带ip）
	 * @param token 密钥
	 * @param secret 用户的密码
	 * @param currentIp 当前请求ip
	 * @return 是否正确
	 */
	public static boolean verify(String token, String secret, String currentIp) {
		try {
			// 根据密码生成JWT效验器
			Algorithm algorithm = Algorithm.HMAC256(secret);
			JWTVerifier verifier = JWT.require(algorithm)
					.withClaim("username", getUsername(token))
					.withClaim("userId", getUserId(token))
					.withClaim("ip", getIP(token))
					.withAudience(audience)
					.build();
			// 效验TOKEN
			verifier.verify(token);
			return true;
		}
		catch (JWTVerificationException exception) {
			return false;
		}
	}

	/**
	 * 校验当前请求的ip和token中的ip是否是同一个
	 * @param token token
	 * @param remoteAddr remoteAddr
	 * @return {@link boolean} ip相同 or token中的ip为空orremoteAddr为空 返回true
	 * @author liaoly
	 * @date 2022/3/31 11:10
	 */
	public static boolean verifyIp(String token, String remoteAddr) {

		String ip = getIP(token);

		if (StringUtils.isBlank(remoteAddr) || StringUtils.isBlank(ip)) {
			return true;
		}

		return remoteAddr.equalsIgnoreCase(ip);
	}

	/**
	 * 校验token是否过期
	 * @author liaoly
	 * @date 2022/3/10 11:32
	 * @param token token
	 * @return {@link Boolean} 过期返回true
	 */
	public static Boolean isTokenExpired(String token) {
		final Date expiration = getExpirationDateFromToken(token);
		return expiration == null || expiration.before(new Date());
	}

	/**
	 * 获取token的过期时间
	 * @author liaoly
	 * @date 2022/3/10 11:32
	 * @param token token
	 * @return {@link Date}
	 */
	private static Date getExpirationDateFromToken(String token) {
		//token 就是jwt生成的token
		try {
			return JWT.decode(token).getClaims().get("exp").asDate();
		} catch (JWTDecodeException e) {
			return null;
		}
	}
}
