package cn.smarthse.radiationTraining.core.framework.logback;

import ch.qos.logback.core.pattern.CompositeConverter;

import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 *
 * logback 打印日志脱敏过滤器
 *      logback.xml文件中引入 转换器，代码如下：
 *     1. {@code <conversionRule conversionWord="masked" converterClass="cn.smarthse.common.framework.log.SensitiveConverter" />}
 *     2. 调用转换器打印日志 %masked(%msg)
 * @author liaoly(廖凌云) [1302013247@qq.com]
 * @date 2023/8/25 15:25
 */
public class SensitiveConverter extends CompositeConverter<Object> {
    /** 手机号 */
    private static final String phonePattern = "(?:0|86|\\+86)?1[3-9]\\d{9}";
    /** 身份证 */
    private static final String idCardPattern = "[1-9]\\d{5}(18|19|20)\\d{2}(0[1-9]|1[0-2])(0[1-9]|1\\d|2[0-9])\\d{3}(\\d|X|x)$|^[1-9]\\d{14}";
    /** 邮箱 */
    private static final String emailPattern = "[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,4}";

    private static final Pattern pattern = Pattern.compile(phonePattern + "|" + idCardPattern + "|" + emailPattern);

    private static final List<Character> prefix = Arrays.asList(':', '：');

    @Override
    protected String transform(Object event, String in) {

        if (in == null) {
            return null;
        }

        // 匹配手机号、身份证号、邮箱的正则表达式
        Matcher matcher = pattern.matcher(in);

        StringBuffer maskedInput = new StringBuffer();
        while (matcher.find()) {
            String matchedString = matcher.group();
            if (matchedString.matches(phonePattern) && extra(matcher, in)) { //匹配手机号 && 前后一位不是数字
                matcher.appendReplacement(maskedInput, maskPartialData(matchedString, 3, 7));
            } else if (matchedString.matches(idCardPattern) && extra(matcher, in)) { //匹配身份证号
                matcher.appendReplacement(maskedInput, maskPartialData(matchedString, 6, 14));
            } else if (matchedString.matches(emailPattern)) { //匹配邮箱
                matcher.appendReplacement(maskedInput, maskEmail(matchedString));
            }
        }

        return matcher.appendTail(maskedInput).toString();
    }

    /**
     * 字符串 in 前后一位不是数字||前后没有字符了 返回true，反之返回false
     * @param matcher
     * @param in
     * @return {@link boolean}
     * @author liaoly(廖凌云) [1302013247@qq.com]
     * @date 2024/5/9 10:18
     */
    private static boolean extra(Matcher matcher, String in) {
        int start = matcher.start();
        int end = matcher.end();
        return  (start == 0 || (!Character.isDigit(in.charAt(start - 1)) && !prefix.contains(in.charAt(start - 1))))
                && (end == in.length() || (!Character.isDigit(in.charAt(end)) && !prefix.contains(in.charAt(end))));
    }

    /**
     * 自定义脱敏 （start 至 end 用 * 替换显示，左闭右开）
     * @param data 数据
     * @param start 开始
     * @param end 结束
     * @return {@link String}
     * @author liaoly
     * @date 2023/8/25 15:27
     */
    public static String maskPartialData(String data, int start, int end) {
        StringBuilder maskedData = new StringBuilder();
        for (int i = 0; i < data.length(); i++) {
            if (i >= start && i < end) {
                maskedData.append('*');
            } else {
                maskedData.append(data.charAt(i));
            }
        }
        return maskedData.toString();
    }

    /**
     * 邮箱脱敏
     * @param email
     * @return {@link String}
     * @author liaoly
     * @date 2023/8/25 15:27
     */
    public static String maskEmail(String email) {
        int atIndex = email.indexOf('@');
        if (atIndex > 0) {
            String username = email.substring(0, atIndex);
            String domain = email.substring(atIndex);
            return maskPartialData(username, 2, username.length()) + domain;
        }
        return email;
    }
    /**
     * 中文姓名脱敏
     * @param name
     * @return {@link String}
     * @author liaoly
     * @date 2023/8/25 15:34
     */
    public static String maskChineseName(String name) {
        if (name.length() > 1) {
            String firstName = name.substring(0, 1);
            String maskedName = maskPartialData(name.substring(1), 0, name.length() - 1);
            return firstName + maskedName;
        }
        return name;
    }

}