package cn.smarthse.radiationTraining.core.framework.utils;

import cn.hutool.core.img.ImgUtil;
import cn.hutool.core.io.IORuntimeException;
import cn.hutool.core.io.IoUtil;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.multipart.MultipartFile;

/**
 *
 * 文件工具类
 *
 * @Author liaoly(廖凌云) [1302013247@qq.com]
 * @Date 2023/7/4 8:51
 */
@Slf4j
public class FileUtil {

    /**
     * 默认缩放比例  1为不缩放 （分辨率不变）
     */
    private static final float scale = 1f;
    /**
     * 触发压缩的最小文件大小 以字节为单位
     */
    private static final long min_size = 102400;

    /**
     * 文件存储到本地 （如果是图片，则会压缩图片）
     * @param inputStream 文件流对象 文件上传直接从 MultipartFile 对象获取
     * @param realPath 输出文件到本地路径，路径不存在会自动创建。 eg：D://upload/radiationTraining/common/202212/2212141631340001.jpg
     * @param scale 缩放比例。比例大于1时为放大，小于1大于0为缩小
     * @author liaoly
     * @date 2023/7/4 9:00
     */
    public static void writeFromStream(InputStream inputStream, String realPath, float scale) {

        if (realPath.endsWith(ImgUtil.IMAGE_TYPE_JPG)
                || realPath.endsWith(ImgUtil.IMAGE_TYPE_JPEG)
                || realPath.endsWith(ImgUtil.IMAGE_TYPE_PNG)
                || realPath.endsWith(ImgUtil.IMAGE_TYPE_BMP)
                || realPath.endsWith(ImgUtil.IMAGE_TYPE_PSD)) {

            FileOutputStream outputStream = null;
            try {
                outputStream = new FileOutputStream(cn.hutool.core.io.FileUtil.touch(realPath));
            } catch (FileNotFoundException e) {
                log.error("FileNotFoundException：", e);
            }

            //压缩图片, 失败则不压缩
            try {
                ImgUtil.scale(inputStream, outputStream, scale);
            } catch (Exception e) {
                log.error("图片压缩异常：{}", e.getMessage());
                cn.hutool.core.io.FileUtil.writeFromStream(inputStream, realPath);
            }

            IoUtil.close(inputStream);
            IoUtil.close(outputStream);

        } else {
            try {
                cn.hutool.core.io.FileUtil.writeFromStream(inputStream, realPath);
            } catch (IORuntimeException e) {
                log.error("IORuntimeException：", e);
            }
        }
    }

    /**
     * 文件存储到本地  （如果是图片，则会不改变分辨率压缩图片）
     * @param inputStream 文件流对象 文件上传直接从 MultipartFile 对象获取
     * @param realPath 输出文件到本地路径，路径不存在会自动创建。 eg：D://upload/radiationTraining/common/202212/2212141631340001.jpg
     * @author liaoly
     * @date 2023/7/4 9:00
     */
    public static void writeFromStream(InputStream inputStream, String realPath) {

        writeFromStream(inputStream, realPath, scale);
    }

    /**
     * 文件存储到本地  （如果是图片 超过一定大小，则会不改变分辨率压缩图片）
     * @param fileData 文MultipartFile 对象
     * @param realPath 输出文件到本地路径，路径不存在会自动创建。 eg：D://upload/radiationTraining/common/202212/2212141631340001.jpg
     * @author liaoly
     * @date 2023/7/4 9:00
     */
    public static void writeFromStream(MultipartFile fileData, String realPath) {

        if (fileData.getSize() <= min_size) {
            try {
                cn.hutool.core.io.FileUtil.writeFromStream(fileData.getInputStream(), realPath);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        try {
            writeFromStream(fileData.getInputStream(), realPath, scale);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取到200kb左右的压缩scale值
     *  一一手动校验值（经验值）
     * @param srcSize 文件大小
     * @return scale
     */
    private static float scale200K(long srcSize) {
        float scale;
        if (srcSize < 200 * 1000) {
            scale = 1.00f;
        } else if (srcSize < 500 * 1000) {
            scale = 0.57f;
        } else if (srcSize < 700 * 1000) {
            scale = 0.47f;
        } else if (srcSize < 1000 * 1000) {
            scale = 0.37f;
        } else if (srcSize < 2 * 1000 * 1000) {
            scale = 0.25f;
        } else if (srcSize < 4 * 1000 * 1000) {
            scale = 0.17f;
        } else if (srcSize < 5 * 1000 * 1000) {
            scale = 0.13f;
        } else if (srcSize < 10 * 1000 * 1000) {
            scale = 0.10f;
        } else {
            scale = 0.01f;
        }
        return scale;
    }
}
