package cn.smarthse.radiationTraining.core.framework.filter;

import org.apache.commons.io.IOUtils;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;

/**
 * 自定义类继承 <code>HttpServletRequestWrapper</code> 实现请求体 <code>RequestBody</code> 的多次读取.
 * <p>
 * 为了解决请求体的 <code>RequestBody</code> 无法多次读取而编写，通过重写 <code>getInputStream</code> 与
 * <code>getReader</code> 方法实现.
 *
 * @author <a href="mailto:xiaoQQya@126.com">xiaoQQya</a>
 * @date 2021/05/06
 * @since 1.0
 */
public class MultiReadHttpServletRequest extends HttpServletRequestWrapper {

    private ByteArrayOutputStream cachedBytes;

    public MultiReadHttpServletRequest(HttpServletRequest request) {
        super(request);
    }

    @Override
    public ServletInputStream getInputStream() throws IOException {
        if (cachedBytes == null) {
            cacheInputStream();
        }
        return new CachedServletInputStream();
    }

    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(getInputStream()));
    }

    /**
     * Cache the inputstream in order to read it multiple times. For
     * convenience, I use apache.commons IOUtils.
     *
     * @throws IOException IOException
     * @author xiaoqqya
     * @date 2021/05/06
     */
    private void cacheInputStream() throws IOException {
        cachedBytes = new ByteArrayOutputStream();
        IOUtils.copy(super.getInputStream(), cachedBytes);
    }

    /**
     * An inputstream which reads the cached request body.
     *
     * @author xiaoqqya
     * @date 2021/05/06
     */
    public class CachedServletInputStream extends ServletInputStream {
        private final ByteArrayInputStream input;

        public CachedServletInputStream() {
            // create a new input stream from the cached request body
            input = new ByteArrayInputStream(cachedBytes.toByteArray());
        }

        @Override
        public boolean isFinished() {
            return false;
        }

        @Override
        public boolean isReady() {
            return false;
        }

        @Override
        public void setReadListener(ReadListener readListener) {

        }

        @Override
        public int read() throws IOException {
            return input.read();
        }
    }
}
