package cn.smarthse.radiationTraining.core.framework.config.desensitization;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.DesensitizedUtil;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.ser.ContextualSerializer;
import java.io.IOException;

/**
 * 序列化过程中 脱敏带有 {@link Desensitization} 注解的String类型的字段
 * 具体脱敏方法 参考 hutool工具类 {@link DesensitizedUtil}
 * @Author liaoly(廖凌云) [1302013247@qq.com]
 * @Date 2023/8/22 14:39
 */
public class DesensitizationSerialize extends JsonSerializer<String> implements ContextualSerializer {

    private DesensitizationType type;

    private Integer start;

    private Integer end;

    public DesensitizationSerialize() {
    }

    public DesensitizationSerialize(DesensitizationType type, int start, int end) {
        this.type  = type;
        this.start = start;
        this.end   = end;
    }

    @Override
    public void serialize(String str, JsonGenerator gen, SerializerProvider serializers) throws IOException {

        switch (type) {
            case CUSTOM_ROLE:
                gen.writeString(CharSequenceUtil.hide(str, start, end));
                break;
            case USER_ID:
                gen.writeString(String.valueOf(DesensitizedUtil.userId()));
                break;
            case CHINESE_NAME:
                gen.writeString(DesensitizedUtil.chineseName(String.valueOf(str)));
                break;
            case ID_CARD:
                gen.writeString(DesensitizedUtil.idCardNum(String.valueOf(str), 1, 2));
                break;
            case FIXED_PHONE:
                gen.writeString(DesensitizedUtil.fixedPhone(String.valueOf(str)));
                break;
            case MOBILE_PHONE:
                gen.writeString(DesensitizedUtil.mobilePhone(String.valueOf(str)));
                break;
            case ADDRESS:
                gen.writeString(DesensitizedUtil.address(String.valueOf(str), 8));
                break;
            case EMAIL:
                gen.writeString(DesensitizedUtil.email(String.valueOf(str)));
                break;
            case PASSWORD:
                gen.writeString(DesensitizedUtil.password(String.valueOf(str)));
                break;
            case CAR_LICENSE:
                gen.writeString(DesensitizedUtil.carLicense(String.valueOf(str)));
                break;
            case BANK_CARD:
                gen.writeString(DesensitizedUtil.bankCard(String.valueOf(str)));
                break;
            default:
        }
    }

    //对于同一个类，例如 PersonalUserVo，返回一次后就不再执行，类字段中有多少Desensitization注解，就执行多少次，只后便不再执行
    @Override
    public JsonSerializer<?> createContextual(SerializerProvider prov, BeanProperty beanProperty) throws JsonMappingException {

        if (beanProperty == null) {
            //如果为BeanProperty空，就用value 为nu11 的序列化方式。
            return prov.findNullValueSerializer(null);
        }

        if (!beanProperty.getType().getRawClass().equals(String.class)) {
            // 如果不是string类型，用系统自带的序列化的方式。
            return prov.findValueSerializer(beanProperty.getType(), beanProperty);
        }

        Desensitization desensitization = beanProperty.getAnnotation(Desensitization.class);
        if (desensitization == null) {
            //如果不存在自定义脱敏注解，也用系统自带的序列化的方式。
            return prov.findValueSerializer(beanProperty.getType(), beanProperty);
        } else {
            return new DesensitizationSerialize(desensitization.type(), desensitization.start(), desensitization.end());
        }
    }
}
