package cn.smarthse.radiationTraining.core.framework.bootstrap;

import cn.smarthse.radiationTraining.core.framework.Constant;
import cn.smarthse.radiationTraining.core.framework.model.ResponseData;
import cn.smarthse.radiationTraining.core.framework.model.ResponseStateEnum;
import cn.smarthse.radiationTraining.core.framework.utils.JwtUtil;
import cn.smarthse.radiationTraining.core.framework.utils.ServletUtils;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.smarthse.radiationTraining.exception.ServiceException;
import javax.validation.ValidationException;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.rpc.RpcException;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authz.AuthorizationException;
import org.apache.shiro.authz.UnauthenticatedException;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MaxUploadSizeExceededException;
import org.springframework.web.servlet.handler.SimpleMappingExceptionResolver;

/**
 *
 * 自定义异常处理 HandlerExceptionResolver接口和@ExceptionHandler注解两者只能二选一，并且，后者优先。
 *
 * @author liaoly(廖凌云) [1302013247@qq.com]
 * @date 2022/4/13 13:42
 */
@ControllerAdvice
@Slf4j
public class SmarthseExceptionHandler extends SimpleMappingExceptionResolver {

	@ExceptionHandler({Exception.class})
	@ResponseBody
	public ResponseData<Object> dealException(Exception ex) {

		ResponseData<Object> r = new ResponseData<>();
		if (ex instanceof BindException || ex instanceof ValidationException) {
			r.setState(ResponseStateEnum.fail);
			List<FieldError> fieldErrors = ((BindException)ex).getBindingResult().getFieldErrors();
			StringBuilder err = new StringBuilder("以下参数错误").append("</br>");
			int index = 1;
			for (FieldError error : fieldErrors) {
				log.error(error.getField() + ":" + error.getDefaultMessage());
				err.append(index++).append(".").append(error.getDefaultMessage()).append("</br>");
			}
			r.setMessage(err.toString());
		} else if (ex instanceof UnauthenticatedException) {
			r.setState(ResponseStateEnum.UNAUTHORIZED);
			r.setMessage("登入失效！");
		} else if (ex instanceof AuthorizationException) {
			r.setState(ResponseStateEnum.Unauthorized403);
			r.setMessage("无相关权限！");
		} else if (ex instanceof AuthenticationException) {
			r.setState(ResponseStateEnum.fail);
			r.setMessage("授权失败，请重新登录");
		} else if (ex instanceof ServiceException) {
			r.setResult(((ServiceException)ex).getResult());
			r.setState(ResponseStateEnum.getState(((ServiceException)ex).getCode()));
			r.setMessage(ex.getMessage());
		} else if (ex instanceof RpcException) {
			r.setState(ResponseStateEnum.INTERNAL_SERVER_ERROR);
			r.setMessage("RPC服务错误！");
		} else if (ex instanceof HttpMessageNotReadableException) {
			r.setState(ResponseStateEnum.INTERNAL_SERVER_ERROR);
			r.setMessage("参数有误！");
		} else if (ex instanceof MaxUploadSizeExceededException) {
			r.setState(ResponseStateEnum.INTERNAL_SERVER_ERROR);
			r.setMessage("文件太大！");
		}else {
			r.setState(ResponseStateEnum.INTERNAL_SERVER_ERROR);
			r.setMessage("操作失败！");
		}

		try {
			HttpServletRequest request = ServletUtils.getRequest();

			log.error( "userId:{}-uri:{}-param:{}-统一捕获异常=>:", JwtUtil.getUserId(request.getHeader(Constant.TOKEN_HEADER_NAME)),
					request.getRequestURI(), ServletUtils.getRequestParam(request), ex);
		} catch (Exception e) {
			log.error("统一捕获异常=>:", ex);
		}

		return r;
	}
}
