package cn.smarthse.radiationTraining.core.framework.bootstrap;


import cn.hutool.json.JSONUtil;
import cn.smarthse.radiationTraining.core.framework.utils.ServletUtils;
import com.alibaba.fastjson.JSON;
import java.util.Objects;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.*;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;

/**
 *
 * 日志切面 可以记录请求的参数，返回值，耗时 异常信息（依赖于配置：spring.profiles.active:dev）
 * “@ConditionalOnProperty” 说明：https://www.cnblogs.com/secbro/p/12011522.html
 * @author liaoly(廖凌云) [1302013247@qq.com]
 * @date 2022/4/12 15:40
 */
@Aspect
@Component
@Slf4j
@ConditionalOnProperty(value = "spring.profiles.active", havingValue = "dev")
@Order(1)
public class LogAopAspect {

	private static final String START_TIME = "request-start";

	private static final String REQUEST_METHOD_GET = "GET";

	/**
	 * 切入点
	 */
	@Pointcut("execution(public * cn.smarthse.radiationTraining..*.controller..*.*(..))")
	public void pointCut() {
	}

	/**
	 * 前置操作
	 * @param point 切入点
	 */
	@Before("pointCut()")
	public void beforeLog(JoinPoint point) {

		HttpServletRequest request = ServletUtils.getRequest();

		try {
			log.debug("【请求URI】：{}-{}", request.getRemoteAddr(), request.getRequestURI());
			//打印请求参数
			printParam(point.getArgs(), request.getRequestURL().toString());

			Long start = System.currentTimeMillis();
			request.setAttribute(START_TIME, start);
		} catch (Exception e) {
			log.error("Exception", e);
		}
	}

	/**
	 * 环绕操作
	 * @param point 切入点
	 * @return 原方法返回值
	 * @throws Throwable 异常信息
	 */
	@Around("pointCut()")
	public Object aroundLog(ProceedingJoinPoint point) throws Throwable {
		Object result = point.proceed();
		log.debug("【请求返回】：{}", JSONUtil.toJsonStr(result));
		return result;
	}

	/**
	 * 后置操作
	 */
	@AfterReturning("pointCut()")
	public void afterReturning() {
		ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
		HttpServletRequest request = Objects.requireNonNull(attributes).getRequest();
		log.debug("【请求耗时】：{}毫秒", System.currentTimeMillis() - (Long) request.getAttribute(START_TIME));
	}

	/**
	 * 是否是常规参数，常规参数可以打印，导入等参数不打印
	 * @param args -
	 * @return {@link boolean}
	 * @author liaoly
	 * @date 2022/3/28 14:01
	 */
	private boolean isCommonParam(Object[] args) {
		for (Object arg : args) {
			//UserEditParam 是标准的编辑操作信息
			if (arg instanceof MultipartFile
				|| arg instanceof HttpServletResponse
				|| arg instanceof HttpServletRequest) {
				return false;
			}
		}
		return true;
	}

	/**
	 *  定义打印请求参数的方法
	 * @param args 参数
	 * @author liaoly
	 * @date 2022/4/12 17:50
	 */
	private void printParam(Object[] args, String url) {

		//是否是常规参数，常规参数可以打印，导入等参数不打印
		if (!isCommonParam(args)) {
			return;
		}

		//登入包含密码等敏感信息不打印
		if (url.contains("/login")) {
			return;
		}

		log.debug("【请求参数】：{}", JSON.toJSONString(args));
	}
}

